export enum StatusColor {
  PRIMARY = '#7C4DFF',
  SUCCESS = '#22C55E',
  WARNING = '#FFAB00',
  ERROR = '#FF5630',
}

export const getBgColor = (color: string, degree = 90) => {
  // degree = pourcentage de blanc ajouté (90 = très clair, 0 = couleur pure)
  if (!color) return '#F5F4FC';

  const hexToRgb = (hex: string) => {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result ? {
      r: parseInt(result[1], 16),
      g: parseInt(result[2], 16),
      b: parseInt(result[3], 16)
    } : null;
  };

  const lightenColor = (color: string, degree: number) => {
    const rgb = hexToRgb(color);
    if (!rgb) return '#F5F4FC';
    const ratio = degree / 100;
    const r = Math.round(rgb.r * (1 - ratio) + 255 * ratio);
    const g = Math.round(rgb.g * (1 - ratio) + 255 * ratio);
    const b = Math.round(rgb.b * (1 - ratio) + 255 * ratio);
    const toHex = (n: number) => n.toString(16).padStart(2, '0');
    return `#${toHex(r)}${toHex(g)}${toHex(b)}`;
  };

  return lightenColor(color, degree);
};


export const calculateTimeSlotWidth = (
  startHour: string,
  startMinute: string,
  endHour: string,
  endMinute: string,
  intervalMinutes: number,
  cellWidth: number
) => {
  const start = parseInt(startHour) * 60 + parseInt(startMinute);
  const end = parseInt(endHour) * 60 + parseInt(endMinute);
  const duration = end - start;
  const intervals = duration / intervalMinutes;
  return intervals * cellWidth - 5;
};

export const groupOverlappingSlots = (slots: any[]) => {
  if (slots.length === 0) return [];

  // Sort slots by start time
  const sortedSlots = [...slots].sort((a, b) => {
    const aStart = parseInt(a.startHeure) * 60 + parseInt(a.startMinute);
    const bStart = parseInt(b.startHeure) * 60 + parseInt(b.startMinute);
    return aStart - bStart;
  });

  const groups: any[][] = [];
  let currentGroup = [sortedSlots[0]];
  let currentEnd = parseInt(sortedSlots[0].endHeure) * 60 + parseInt(sortedSlots[0].endMinute);

  for (let i = 1; i < sortedSlots.length; i++) {
    const slot = sortedSlots[i];
    const slotStart = parseInt(slot.startHeure) * 60 + parseInt(slot.startMinute);
    const slotEnd = parseInt(slot.endHeure) * 60 + parseInt(slot.endMinute);

    if (slotStart < currentEnd) {
      // This slot overlaps with the current group
      currentGroup.push(slot);
      currentEnd = Math.max(currentEnd, slotEnd);
    } else {
      // No overlap, start a new group
      groups.push(currentGroup);
      currentGroup = [slot];
      currentEnd = slotEnd;
    }
  }

  // Don't forget to add the last group
  groups.push(currentGroup);
  return groups;
};

export const findSlotIndexInGroup = (slot: any, groups: any[][]) => {
  for (const group of groups) {
    const index = group.findIndex(s => s.id === slot.id);
    if (index !== -1) {
      return index;
    }
  }
  return 0;
};

// Add other utility functions as needed 