import { 
  faClock, 
  faCheckCircle, 
  faTimesCircle, 
  faListCheck
} from '@fortawesome/free-solid-svg-icons';
import PieCharts from '@/shared/components/charts/pie-charts';
import StatsCards from '@/shared/components/charts/stats-cards';
import { Box } from '@mui/material';
import { IDemandeAbsence } from '@/shared/types/leave';

interface CongeOverviewProps {
  data: IDemandeAbsence[];
}

export default function CongeOverview({ data }: CongeOverviewProps) {
  // Calculate real statistics from the data
  const totalDemandes = data.length;
  const demandesEnAttente = data.filter(item => 
    item.status === 'En attente' || item.status === 'Waiting' || item.status === 'Unprocessed'
  ).length;
  const demandesValidees = data.filter(item => 
    item.status === 'Approuvée'
  ).length;
  const demandesRefusees = data.filter(item => 
    item.status === 'Refusée'
  ).length;

  // Calculate leave type statistics
  const leaveTypeCounts = data.reduce((acc, item) => {
    const type = item.leaveType;
    acc[type] = (acc[type] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);

  const typeStats = [
    { label: 'Maladie', count: leaveTypeCounts['LEAVE_SICK'] || 0 },
    { label: 'Parental', count: leaveTypeCounts['LEAVE_PARENTAL'] || 0 }, 
    { label: 'Annuel', count: leaveTypeCounts['LEAVE_PAID'] || 0 },
    { label: 'RTT', count: leaveTypeCounts['LEAVE_RTT'] || 0 }, 
    { label: 'Sans solde', count: leaveTypeCounts['LEAVE_UNPAID'] || 0 },
  ];

  const stats = [
    {
      title: 'Total Demandes',
      total: totalDemandes,
      icon: faListCheck,
      color: '#5176FE',
    },
    {
      title: 'En Attente',
      total: demandesEnAttente,
      icon: faClock,
      color: '#F9B90B',
    },
    {
      title: 'Validées',
      total: demandesValidees,
      icon: faCheckCircle,
      color: '#36B37E',
    },
    {
      title: 'Refusées',
      total: demandesRefusees,
      icon: faTimesCircle,
      color: '#FF3434',
    },
  ];

  // Calculate percentages for charts
  const statusChartData = totalDemandes > 0 ? [
    { 
      id: 'En attente', 
      value: demandesEnAttente, 
      label: `En attente (${demandesEnAttente}) ${Math.round((demandesEnAttente / totalDemandes) * 100)}%`, 
      color: '#F9B90B' 
    },
    { 
      id: 'Validé', 
      value: demandesValidees, 
      label: `Validé (${demandesValidees}) ${Math.round((demandesValidees / totalDemandes) * 100)}%`, 
      color: '#36B37E' 
    },
    { 
      id: 'Refusé', 
      value: demandesRefusees, 
      label: `Refusé (${demandesRefusees}) ${Math.round((demandesRefusees / totalDemandes) * 100)}%`, 
      color: '#FF3434' 
    },
  ] : [];

  const typeChartData = totalDemandes > 0 ? typeStats.map((stat, index) => {
    const colors = ['#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFB6C1'];
    const percentage = Math.round((stat.count / totalDemandes) * 100);
    return {
      id: stat.label,
      value: stat.count,
      label: `${stat.label} (${stat.count}) ${percentage}%`,
      color: colors[index],
    };
  }).filter(item => item.value > 0) : []; // Only show types with data

  const charts = [
    {
      title: 'Répartition par statut',
      data: statusChartData,
    },
    {
      title: 'Répartition par type',
      data: typeChartData,
    },
  ].filter(chart => chart.data.length > 0); // Only show charts with data

  return (
    <Box sx={{ width: '100%' }}>
      <PieCharts charts={charts} />
      <StatsCards stats={stats} />
    </Box>
  );
}