'use client';

import { useState, useEffect } from 'react';
import {
  IconButton,
  Stack,
  Box,
  Slider,
  Typography,
} from '@mui/material';
import FontAwesome from '@/shared/components/fontawesome';
import { WhiteIconButtonStyle } from '@/shared/theme/css';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import TableFilter from '@/shared/components/table/table-filter';
import {
  faCircleChevronLeft,
  faCircleChevronRight,
  faPrint,
  faRotate,
  faAngleDoubleLeft,
} from '@fortawesome/free-solid-svg-icons';
import { alpha, styled } from '@mui/system';
import { mainButtonStyle } from '@/shared/components/table/styles';
import { TableColumn, TableType } from '@/shared/types/common';
import { toolbarStyles } from '../stytles';


const CustomSlider = styled(Slider)(({ theme }) => ({
  color: theme.palette.primary.main,
  height: 4,
  '& .MuiSlider-thumb': {
    height: 16,
    width: 16,
    backgroundColor: '#fff',
    border: `6px solid ${theme.palette.primary.main}`,
    boxShadow: `0 0 0 3px white`,
    '&:hover, &.Mui-focusVisible': {
      boxShadow: `0 0 0 8px ${alpha(theme.palette.primary.main, 0.16)}`,
    },
  },
  '& .MuiSlider-track': {
    height: 1,
    borderRadius: 2,
  },
  '& .MuiSlider-rail': {
    height: 3,
    borderRadius: 2,
    backgroundColor: alpha(theme.palette.primary.main, 0.2),
  },
}));

const ExtendedWhiteIconButtonStyle = {
  ...WhiteIconButtonStyle,
  padding: 0.75,
};

interface ScheduleToolbarProps {
  date: Date;
  currentView: string;
  onViewChange: (view: string) => void;
  onDateChange: (date: Date) => void;
  onZoomChange: (zoom: number) => void;
  onFilters: (key: string, value: string, condition: string) => void;
  onResetFilters: () => void;
  onDrawerToggle: () => void;
  onNavigate: (direction: 'prev' | 'next') => void;
  tableHead: TableColumn[];
  zoom?: number;
}

export function ScheduleToolbar({
  currentView,
  onZoomChange,
  onFilters,
  onResetFilters,
  onDrawerToggle,
  onNavigate,
  tableHead,
  zoom = 90,
}: ScheduleToolbarProps) {
  const [zoomValue, setZoomValue] = useState<number>(zoom);

  useEffect(() => {
    setZoomValue(zoom);
  }, [zoom]);

  const getNavigationTitle = () => {
    switch(currentView) {
      case 'dayGridMonth':
        return 'mois';
      case 'timeGridWeek':
        return 'semaine';
      case 'timeGridDay':
        return 'jour';
      default:
        return 'période';
    }
  };

  const handleZoomChange = (_event: Event, newValue: number | number[]) => {
    const zoomLevel = Array.isArray(newValue) ? newValue[0] : newValue;
    setZoomValue(zoomLevel);

    const baseHeight = 50; 
    const slotHeight = Math.floor((zoomLevel / 100) * baseHeight);
    const timegridEventHeight = Math.max(30, Math.floor(slotHeight * 0.7)); // CORRIGÉ: Hauteur minimale plus élevée
    const daygridEventHeight = Math.max(25, Math.floor(slotHeight / 2));

    document.documentElement.style.setProperty('--fc-slot-height', `${slotHeight}px`);
    document.documentElement.style.setProperty('--fc-timegrid-event-height', `${timegridEventHeight}px`); // NOUVEAU
    document.documentElement.style.setProperty('--fc-daygrid-event-height', `${daygridEventHeight}px`);
    
    onZoomChange(zoomLevel);
    };

  return (
    <Box sx={toolbarStyles.container}>
      <Stack direction="row" spacing={1} alignItems="center">
        <CustomTooltip title="Afficher la liste" arrow>
          <IconButton 
            onClick={onDrawerToggle}
            sx={ExtendedWhiteIconButtonStyle}
          >
            <FontAwesome icon={faAngleDoubleLeft} width={18} />
          </IconButton>
        </CustomTooltip>

        <CustomTooltip title="Imprimer" arrow>
          <IconButton sx={ExtendedWhiteIconButtonStyle}>
            <FontAwesome icon={faPrint} width={18} />
          </IconButton>
        </CustomTooltip>

        <CustomTooltip title={`${getNavigationTitle()} précédent`} arrow>
          <IconButton 
            sx={mainButtonStyle} 
            onClick={() => onNavigate('prev')}
          >
            <FontAwesome width={30} icon={faCircleChevronLeft} />
          </IconButton>
        </CustomTooltip>

        <Box sx={toolbarStyles.sliderContainer}>
          <CustomSlider
            value={zoomValue}
            onChange={handleZoomChange}
            aria-labelledby="continuous-slider"
            min={25}
            max={200}
            step={5}
          />
          <Typography sx={toolbarStyles.sliderValue}>
            {zoomValue}%
          </Typography>
        </Box>

        <CustomTooltip title={`${getNavigationTitle()} suivant`} arrow>
          <IconButton 
            sx={mainButtonStyle} 
            onClick={() => onNavigate('next')}
          >
            <FontAwesome width={30} icon={faCircleChevronRight} />
          </IconButton>
        </CustomTooltip>

        <CustomTooltip title="Rafraîchir" arrow>
          <IconButton 
            sx={ExtendedWhiteIconButtonStyle} 
            onClick={onResetFilters}
          >
            <FontAwesome icon={faRotate} width={18} />
          </IconButton>
        </CustomTooltip>

        <TableFilter
          onFilters={onFilters}
          onResetFilters={onResetFilters}
          initialColumns={tableHead}
          type={TableType.Appointment}
        />
      </Stack>
    </Box>
  );
}