import { IBesoinRecrutement, BesoinRecrutementStatut } from '@/shared/types/besoin-recrutement';
import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { 
  _besoinsRecrutement, 
  DEFAULT_BESOIN_RECRUTEMENT_TABLE_HEAD,
  DEFAULT_BESOIN_RECRUTEMENT_EN_ATTENTE_TABLE_HEAD,
  DEFAULT_BESOIN_RECRUTEMENT_EN_COURS_TABLE_HEAD,
  DEFAULT_BESOIN_RECRUTEMENT_TRAITES_TABLE_HEAD,
  DEFAULT_BESOIN_RECRUTEMENT_TOUS_TABLE_HEAD
} from '@/shared/_mock/_besoin-recrutement';
import { BesoinRecrutementTab } from '@/shared/types/besoin-recrutement';
import { useMemo, useCallback, useEffect } from 'react';

interface BesoinRecrutementTableConfig extends TableConfig<IBesoinRecrutement> {
  activeTab: BesoinRecrutementTab;
}

export function useBesoinRecrutementTable(activeTab: BesoinRecrutementTab) {
  const getTableHead = useMemo(() => {
    switch (activeTab) {
      case BesoinRecrutementTab.EnAttente:
        return DEFAULT_BESOIN_RECRUTEMENT_EN_ATTENTE_TABLE_HEAD;
      case BesoinRecrutementTab.EnCours:
        return DEFAULT_BESOIN_RECRUTEMENT_EN_COURS_TABLE_HEAD;
      case BesoinRecrutementTab.Traites:
        return DEFAULT_BESOIN_RECRUTEMENT_TRAITES_TABLE_HEAD;
      case BesoinRecrutementTab.Tous:
        return DEFAULT_BESOIN_RECRUTEMENT_TOUS_TABLE_HEAD;
      default:
        return DEFAULT_BESOIN_RECRUTEMENT_TABLE_HEAD;
    }
  }, [activeTab]);

  // Filter data based on active tab
  const filteredInitialData = useMemo(() => {
    switch (activeTab) {
      case BesoinRecrutementTab.EnAttente:
        return _besoinsRecrutement.filter(besoin => besoin.statut === BesoinRecrutementStatut.EnAttente);
      case BesoinRecrutementTab.EnCours:
        return _besoinsRecrutement.filter(besoin => besoin.statut === BesoinRecrutementStatut.EnCours);
      case BesoinRecrutementTab.Traites:
        return _besoinsRecrutement.filter(besoin => besoin.statut === BesoinRecrutementStatut.Traite);
      case BesoinRecrutementTab.Tous:
        return _besoinsRecrutement;
      default:
        return _besoinsRecrutement;
    }
  }, [activeTab]);

  const besoinRecrutementConfig: BesoinRecrutementTableConfig = {
    initialData: filteredInitialData,
    defaultTableHead: getTableHead,
    activeTab
  };

  const tableManager = useTableManager<IBesoinRecrutement>(besoinRecrutementConfig);
  
  // Force update the table head when the active tab changes
  useEffect(() => {
    tableManager.handleResetColumns();
  }, [activeTab, tableManager]);
  
  // Custom handleSave function to update the global _besoinsRecrutement array
  const handleSave = useCallback((updatedBesoinRecrutement: IBesoinRecrutement) => {
    // Update the global _besoinsRecrutement array
    const index = _besoinsRecrutement.findIndex(b => b.id === updatedBesoinRecrutement.id);
    if (index !== -1) {
      _besoinsRecrutement[index] = updatedBesoinRecrutement;
    } else {
      _besoinsRecrutement.push(updatedBesoinRecrutement);
    }
    
    // Return the updated item
    return updatedBesoinRecrutement;
  }, []);

  return {
    ...tableManager,
    handleSave,
  };
} 