import * as Yup from "yup"
import { CandidatureStatut } from "@/shared/types/candidature"

// Schéma de validation pour les candidatures
export const createCandidatureSchema = (isSedentaire: boolean) => {
  return Yup.object().shape({
    // Champs communs
    nom: Yup.string().required("Le nom est obligatoire"),
    prenom: Yup.string().required("Le prénom est obligatoire"),
    email: Yup.string().email("Format d'email invalide").required("L'email est obligatoire"),
    email_pro: Yup.string().email("Format d'email professionnel invalide").nullable(),
    telephone: Yup.string()
      .matches(/^(0|\+33)[1-9]([-. ]?[0-9]{2}){4}$/, "Format de téléphone invalide (ex: 06 12 34 56 78)")
      .required("Le téléphone est obligatoire"),
    age: Yup.number().min(16, "L'âge minimum est de 16 ans").max(80, "L'âge maximum est de 80 ans").nullable(),
    codePostal: Yup.string()
      .matches(/^[0-9]{5}$/, "Le code postal doit contenir 5 chiffres")
      .required("Le code postal est obligatoire"),
    ville: Yup.string().nullable(),
    departement: Yup.string().nullable(), // Plus obligatoire - champ de texte libre
    statut: Yup.string().required("Le statut est obligatoire"),

    // Champs conditionnels
    dateHeureRdvAgence: Yup.string().when("statut", {
      is: (statut: string) => statut === CandidatureStatut.Retenu || statut === CandidatureStatut.Valide,
      then: () => Yup.string().required("Date de RDV obligatoire pour ce statut"),
      otherwise: () => Yup.string().nullable(),
    }),

    // Suivi d'appel - plus obligatoire
    suiviAppel: Yup.string().nullable(),

    // Date appel et rappel - plus obligatoire
    dateAppelRappel: Yup.string().nullable(),

    // Disponibilité - plus obligatoire
    disponibilite: Yup.string().nullable(),

    raisonRefus: Yup.string().when("statut", {
      is: (statut: string) => statut === CandidatureStatut.Refuse || statut === CandidatureStatut.Blackliste,
      then: () => Yup.string().required("Raison du refus obligatoire pour ce statut"),
      otherwise: () => Yup.string().nullable(),
    }),

    // Proximité de circuit - plus obligatoire
    proximiteCircuit: Yup.string().nullable(),

    address: Yup.object().when("statut", {
      is: (statut: CandidatureStatut) => [CandidatureStatut.Refuse, CandidatureStatut.Blackliste, CandidatureStatut.Retenu, CandidatureStatut.Valide, CandidatureStatut.ATraiter].includes(statut),
      then: () => Yup.object().shape({
        address: Yup.string().required("L'adresse est obligatoire"),
        lat: Yup.number().nullable(),
        lng: Yup.number().nullable(),
      }),
      otherwise: () => Yup.object().nullable(),
    }),

    // Durée de permis - obligatoire pour les chauffeurs
    dureePermis: Yup.string().when("type", {
      is: (type: string) => type === "Candidature chauffeur",
      then: () => Yup.string()
        .matches(/^[0-9]+$/, "La durée de permis doit être un nombre")
        .required("La durée de permis est obligatoire pour les chauffeurs"),
      otherwise: () => Yup.string().nullable(),
    }),

    sourceCandidat: Yup.string().nullable(),
    dateCandidature: Yup.string().nullable(),
    document: Yup.string().nullable(),
    contrat: Yup.string().nullable(),
    contrat_document: Yup.string().nullable(),
  })
}
