"use client"

import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  Divider,
  Chip,
  Grid,
  Paper,
  IconButton,
} from "@mui/material"
import {
  Close as CloseIcon,
  Folder as FolderIcon,
  Person as PersonIcon,
  Storage as StorageIcon,
  Category as CategoryIcon,
  Star as StarIcon,
  StarBorder as StarBorderIcon,
  CalendarToday as CalendarIcon,
} from "@mui/icons-material"
import FileThumbnail from "@/shared/components/file-thumbnail"
import type { DocumentDTO } from "@/shared/types/document"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"

interface DocumentDetailsDialogProps {
  open: boolean
  onClose: () => void
  document: DocumentDTO | null
}

const formatFileSize = (bytes: number): string => {
  if (bytes === 0) return "0 Bytes"
  const k = 1024
  const sizes = ["Bytes", "KB", "MB", "GB"]
  const i = Math.floor(Math.log(bytes) / Math.log(k))
  return Number.parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i]
}

const formatDocumentType = (type: string): string => {
  const typeMap: Record<string, string> = {
    WORK_CERTIFICATE: "Certificat de travail",
    PRESENCE_CERTIFICATE: "Certificat de présence",
    ID_CARD: "Carte d'identité",
    DRIVER_LICENCE: "Permis de conduire",
  }
  return typeMap[type] || type
}

export default function DocumentDetailsDialog({
  open,
  onClose,
  document,
}: DocumentDetailsDialogProps) {
  if (!document) return null

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <DialogTitle>
        <Box sx={{ display: "flex", alignItems: "center", justifyContent: "space-between" }}>
          <Box sx={{ display: "flex", alignItems: "center", gap: 2 }}>
            <FileThumbnail file={document.contentType || "PDF"} sx={{ width: 40, height: 40 }} />
            <Box>
              <Typography variant="h6" component="div">
                {document.name}
              </Typography>
              <Typography variant="body2" color="text.secondary">
                Détails du document
              </Typography>
            </Box>
          </Box>
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <ConditionalComponent isValid={document.favorite}>
              <StarIcon color="warning" />
            </ConditionalComponent>
            <IconButton onClick={onClose} size="small">
              <CloseIcon />
            </IconButton>
          </Box>
        </Box>
      </DialogTitle>

      <DialogContent>
        <Grid container spacing={3}>
          {/* Basic Information */}
          <Grid item xs={12}>
            <Paper variant="outlined" sx={{ p: 2 }}>
              <Typography variant="subtitle1" gutterBottom sx={{ fontWeight: 600 }}>
                Informations générales
              </Typography>
              <Divider sx={{ mb: 2 }} />
              
              <Grid container spacing={2}>
                <Grid item xs={12} sm={6}>
                  <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1 }}>
                    <StorageIcon fontSize="small" color="action" />
                    <Typography variant="body2" color="text.secondary">
                      Taille du fichier
                    </Typography>
                  </Box>
                  <Typography variant="body1" sx={{ ml: 3 }}>
                    {formatFileSize(document.size)}
                  </Typography>
                </Grid>

                <Grid item xs={12} sm={6}>
                  <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1 }}>
                    <CategoryIcon fontSize="small" color="action" />
                    <Typography variant="body2" color="text.secondary">
                      Type de contenu
                    </Typography>
                  </Box>
                  <Chip 
                    label={document.contentType} 
                    size="small" 
                    variant="outlined"
                    sx={{ ml: 3 }}
                  />
                </Grid>

                <Grid item xs={12} sm={6}>
                  <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1 }}>
                    <CategoryIcon fontSize="small" color="action" />
                    <Typography variant="body2" color="text.secondary">
                      Type de document
                    </Typography>
                  </Box>
                  <Typography variant="body1" sx={{ ml: 3 }}>
                    {formatDocumentType(document.documentType)}
                  </Typography>
                </Grid>

                <Grid item xs={12} sm={6}>
                  <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1 }}>
                    <PersonIcon fontSize="small" color="action" />
                    <Typography variant="body2" color="text.secondary">
                      Propriétaire
                    </Typography>
                  </Box>
                  <Typography variant="body1" sx={{ ml: 3 }}>
                    ID: {document.ownerId}
                  </Typography>
                </Grid>
              </Grid>
            </Paper>
          </Grid>

          {/* Location Information */}
          <Grid item xs={12}>
            <Paper variant="outlined" sx={{ p: 2 }}>
              <Typography variant="subtitle1" gutterBottom sx={{ fontWeight: 600 }}>
                Emplacement
              </Typography>
              <Divider sx={{ mb: 2 }} />
              
              <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1 }}>
                <FolderIcon fontSize="small" color="action" />
                <Typography variant="body2" color="text.secondary">
                  Dossier parent
                </Typography>
              </Box>
              <Typography variant="body1" sx={{ ml: 3 }}>
                {document.folder ? document.folder.name : "Racine"}
              </Typography>

              <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1, mt: 2 }}>
                <StorageIcon fontSize="small" color="action" />
                <Typography variant="body2" color="text.secondary">
                  Chemin de stockage
                </Typography>
              </Box>
              <Typography 
                variant="body2" 
                sx={{ 
                  ml: 3, 
                  fontFamily: 'monospace', 
                  backgroundColor: 'grey.100', 
                  p: 1, 
                  borderRadius: 1,
                  wordBreak: 'break-all'
                }}
              >
                {document.path}
              </Typography>
            </Paper>
          </Grid>

          {/* Technical Information */}
          <Grid item xs={12}>
            <Paper variant="outlined" sx={{ p: 2 }}>
              <Typography variant="subtitle1" gutterBottom sx={{ fontWeight: 600 }}>
                Informations techniques
              </Typography>
              <Divider sx={{ mb: 2 }} />
              
              <Grid container spacing={2}>
                <Grid item xs={12}>
                  <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1 }}>
                    <StorageIcon fontSize="small" color="action" />
                    <Typography variant="body2" color="text.secondary">
                      Clé S3
                    </Typography>
                  </Box>
                  <Typography 
                    variant="body2" 
                    sx={{ 
                      ml: 3, 
                      fontFamily: 'monospace', 
                      backgroundColor: 'grey.100', 
                      p: 1, 
                      borderRadius: 1,
                      wordBreak: 'break-all'
                    }}
                  >
                    {document.s3Key}
                  </Typography>
                </Grid>

                <Grid item xs={12}>
                  <Box sx={{ display: "flex", alignItems: "center", gap: 1, mb: 1 }}>
                    <PersonIcon fontSize="small" color="action" />
                    <Typography variant="body2" color="text.secondary">
                      UUID du propriétaire
                    </Typography>
                  </Box>
                  <Typography 
                    variant="body2" 
                    sx={{ 
                      ml: 3, 
                      fontFamily: 'monospace', 
                      backgroundColor: 'grey.100', 
                      p: 1, 
                      borderRadius: 1,
                      wordBreak: 'break-all'
                    }}
                  >
                    {document.ownerUuid}
                  </Typography>
                </Grid>
              </Grid>
            </Paper>
          </Grid>
        </Grid>
      </DialogContent>

      <DialogActions>
        <Button onClick={onClose} variant="outlined">
          Fermer
        </Button>
      </DialogActions>
    </Dialog>
  )
}