"use client"

import { useState } from "react"
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  Radio,
  FormControlLabel,
  Divider,
} from "@mui/material"
import { Folder as FolderIcon, Home as HomeIcon } from "@mui/icons-material"
import type { FolderDTO, DocumentDTO } from "@/shared/types/document"
import FolderTreeNode from "./FolderTreeNode"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"

interface MoveDocumentDialogProps {
  open: boolean
  onClose: () => void
  onConfirm: (folderId: number) => void
  folders: FolderDTO[]
  document: DocumentDTO | null
  currentFolderId?: number | null
}

export default function MoveDocumentDialog({
  open,
  onClose,
  onConfirm,
  folders,
  document,
  currentFolderId,
}: MoveDocumentDialogProps) {
  const [selectedFolderId, setSelectedFolderId] = useState<number>(0)

  const handleConfirm = () => {
    onConfirm(selectedFolderId)
    onClose()
  }

  const handleSelectRoot = () => {
    setSelectedFolderId(0)
  }

  const handleSelectFolder = (folderId: number) => {
    setSelectedFolderId(folderId)
  }

  if (!document) return null

  return (
    <Dialog open={open} onClose={onClose} maxWidth="sm" fullWidth>
      <DialogTitle>
        <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
          <FolderIcon color="primary" />
          <Typography variant="h6">
            Déplacer &quot;{document.name}&quot;
          </Typography>
        </Box>
      </DialogTitle>

      <DialogContent>
        <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
          Sélectionnez le dossier de destination pour ce document.
        </Typography>

        <Box sx={{ maxHeight: 400, overflowY: "auto" }}>
          {/* Root option */}
          <Box sx={{ display: "flex", alignItems: "center", mb: 2 }}>
            <HomeIcon sx={{ color: "primary.main", mr: 1 }} fontSize="small" />
            <FormControlLabel
              control={
                <Radio
                  checked={selectedFolderId === null}
                  onChange={handleSelectRoot}
                  disabled={currentFolderId === null}
                />
              }
              label={
                <Typography variant="body2">
                  Racine {currentFolderId === null && "(emplacement actuel)"}
                </Typography>
              }
            />
          </Box>

          <Divider sx={{ mb: 2 }} />

          {/* Folder tree */}
          <ConditionalComponent isValid={folders.length > 0}>
            {
              folders.map( (folder) => (
                <FolderTreeNode
                key={folder.id}
                folder={folder}
                selectedId={selectedFolderId}
                onSelect={handleSelectFolder}
                currentFolderId={currentFolderId || -1}
              />
              ))
            }
          </ConditionalComponent>
          <ConditionalComponent isValid={folders.length === 0}>
            <Typography variant="body2" color="text.secondary" sx={{ textAlign: "center", py: 2 }}>
              Aucun dossier disponible
            </Typography>
          </ConditionalComponent>
        </Box>
      </DialogContent>

      <DialogActions>
        <Button onClick={onClose} color="inherit">
          Annuler
        </Button>
        <Button 
          onClick={handleConfirm} 
          variant="contained"
          disabled={selectedFolderId === currentFolderId || (selectedFolderId === null && currentFolderId === null)}
        >
          Déplacer
        </Button>
      </DialogActions>
    </Dialog>
  )
}