import React, { FC } from 'react';
import {
  Card,
  CardContent,
  Avatar,
  Typography,
  List,
  ListItem,
  ListItemIcon,
  ListItemText,
  Box,
  alpha,
  Chip,
} from '@mui/material';
import EmailIcon from '@mui/icons-material/Email';
import PhoneIcon from '@mui/icons-material/Phone';
import BusinessIcon from '@mui/icons-material/Business';
import DepartmentIcon from '@mui/icons-material/AccountTree';
import LocationOnIcon from '@mui/icons-material/LocationOn';
import { IAgent } from '@/shared/types/client';
import FontAwesome from '@/shared/components/fontawesome';
import { faCalendar } from '@fortawesome/free-regular-svg-icons';

interface Infoprops {
  collaborator: IAgent | null;
  open: () => void;
}

const UserProfileCard: FC<Infoprops> = ({ collaborator, open }) => {
  const getInitials = (firstName: string, lastName: string) => {
    if (!firstName && !lastName) return 'NA';
    const first = firstName ? firstName.charAt(0) : '';
    const last = lastName ? lastName.charAt(0) : '';
    return (first + last).toUpperCase();
  };

  const getDisplayRole = () => {
    if (!collaborator?.roles || collaborator.roles.length === 0) return 'Collaborateur';

    // Filtrer les rôles pour exclure "collaborator" et prendre le deuxième si disponible
    const filteredRoles = collaborator.roles.filter(role =>
      role.name?.toLowerCase() !== 'collaborator'
    );

    if (filteredRoles.length === 0) return 'Collaborateur';

    // Retourner le deuxième rôle s'il existe, sinon le premier
    return filteredRoles[1]?.name || filteredRoles[0]?.name || 'Collaborateur';
  };

  if (!collaborator) {
    return (
      <Card sx={{ borderRadius: 2, boxShadow: 2, height: '100%', display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
        <Typography variant="body1" color="text.secondary">
          Chargement des informations du collaborateur...
        </Typography>
      </Card>
    );
  }

  return (
    <Card
      sx={{
        borderRadius: 2,
        boxShadow: 2,
        height: '100%',
        background: 'linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%)',
      }}
    >
      <CardContent sx={{ p: 3, height: '100%' }}>
        <Box sx={{ display: 'flex', flexDirection: 'column', height: '100%' }}>
          {/* En-tête avec avatar et informations principales */}
          <Box sx={{ display: 'flex', alignItems: 'flex-start', mb: 3 }}>
            <Avatar
              src={collaborator.photoUrl || ''}
              sx={{
                bgcolor: 'primary.main',
                width: 120,
                height: 120,
                mr: 3,
                fontSize: '2.5rem',
                border: '4px solid white',
                boxShadow: 3
              }}
            >
              {getInitials(collaborator.firstName, collaborator.lastName)}
            </Avatar>

            <Box sx={{ flex: 1 }}>
              <Typography variant="h3" component="h1" fontWeight="bold" color="primary" gutterBottom>
                {collaborator.firstName} {collaborator.lastName}
              </Typography>


              <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3, alignItems: 'flex-start' }}>
                
                <Chip
                  label={getDisplayRole()}
                  color="primary"
                  variant="filled"
                  sx={{
                    fontSize: '1rem',
                    height: 32,
                    px: 2,
                    fontWeight: 'medium',
                    width: 'fit-content'
                  }}
                />


                <Chip
                  onClick={open}
                  icon={<FontAwesome icon={faCalendar} width={16} />}
                  label="Votre emploi du temps"
                  sx={{
                    cursor: 'pointer',
                    bgcolor: alpha('#a8a1f7', 0.3),
                    color: 'primary.main',
                    fontSize: '1rem',
                    height: 32,
                    px: 2,
                    fontWeight: 'medium',
                    width: 'fit-content',
                    '&:hover': {
                      bgcolor: alpha('#a8a1f7', 0.5),
                    }
                  }}
                />
              </Box>
            </Box>

            {/* Informations de contact en haut à droite */}
            <Box sx={{
              minWidth: 350,
              ml: 2
            }}>
              <Typography variant="h6" fontWeight="600" color="primary" gutterBottom>
                Informations de contact
              </Typography>

              <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
                {/* Email */}
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                  <EmailIcon color="primary" fontSize="small" sx={{ mr: 1 }} />
                  <Box>
                    <Typography variant="caption" color="text.secondary" display="block">
                      E-mail
                    </Typography>
                    <Typography variant="body2" fontWeight="medium">
                      {collaborator.email || 'N/A'}
                    </Typography>
                  </Box>
                </Box>

                {/* Téléphone */}
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                  <PhoneIcon color="primary" fontSize="small" sx={{ mr: 1 }} />
                  <Box>
                    <Typography variant="caption" color="text.secondary" display="block">
                      Téléphone
                    </Typography>
                    <Typography variant="body2" fontWeight="medium">
                      {collaborator.phoneNumber || 'N/A'}
                    </Typography>
                  </Box>
                </Box>

                {/* Lieu */}
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                  <LocationOnIcon color="primary" fontSize="small" sx={{ mr: 1 }} />
                  <Box>
                    <Typography variant="caption" color="text.secondary" display="block">
                      Lieu
                    </Typography>
                    <Typography variant="body2" fontWeight="medium">
                      {collaborator.addresses?.[0]?.city || 'N/A'}
                    </Typography>
                  </Box>
                </Box>
              </Box>
            </Box>
          </Box>
        </Box>
      </CardContent>
    </Card>
  );
};

export default UserProfileCard;
