"use client"

import { Dialog, DialogTitle, DialogContent, IconButton, CircularProgress, Alert, Box } from "@mui/material"
import { Close as CloseIcon } from "@mui/icons-material"
import { useFormik } from "formik"
import { useState } from "react"
import type { EstablishmentHoursResponse, EstablishmentRequest, UpdateEstablishmentHoursRequest } from "@/shared/types/establishment"
import EstablishmentHoursForm from "@/shared/sections/establishment/establishmentList/components/establishment-hours-form"
import { convertHoursResponseToRequest } from "./establishment-hours-converter"
import { useEstablishmentStore } from "@/shared/api/stores/admin-service/establishmentStore"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import { enqueueSnackbar } from "notistack"

interface UpdateEstablishmentHoursModalProps {
    open: boolean
    onClose: () => void
    establishmentId: number
    establishmentName: string
    initialHours: EstablishmentHoursResponse[]
    onHoursUpdated: () => void
}

export default function UpdateEstablishmentHoursModal({
    open,
    onClose,
    establishmentId,
    establishmentName,
    initialHours,
    onHoursUpdated,
}: UpdateEstablishmentHoursModalProps) {
    const { updateEstablishmentHours, establishmentHoursLoading, error } = useEstablishmentStore()

    const formik = useFormik<EstablishmentRequest>({
        initialValues: convertHoursResponseToRequest(initialHours),
        onSubmit: async (values) => {
            try {
                await updateEstablishmentHours(establishmentId.toString(), values as UpdateEstablishmentHoursRequest)
                onHoursUpdated()
                enqueueSnackbar('Horaires mis à jour avec succès', { variant: 'success' })
                onClose()
            } catch (error: any) {
                enqueueSnackbar(error.message, { variant: 'error' })
            }
        },
        enableReinitialize: true,
    })

    const handleClose = (isSaved: boolean) => {
        if (!isSaved) {
            formik.resetForm()
        }
        onClose()
    }

    return (
        <Dialog
            open={open}
            onClose={() => handleClose(false)}
            maxWidth="lg"
            fullWidth
            PaperProps={{
                sx: {
                    maxHeight: "90vh",
                },
            }}
        >
            <DialogTitle
                sx={{
                    display: "flex",
                    justifyContent: "space-between",
                    alignItems: "center",
                    color: "primary.main",
                    fontWeight: "bold",
                    fontSize: "1.5rem",
                }}
            >
                Horaires de {establishmentName}
                <IconButton
                    aria-label="close"
                    onClick={() => handleClose(false)}
                    sx={{
                        color: (theme) => theme.palette.grey[500],
                    }}
                >
                    <CloseIcon />
                </IconButton>
            </DialogTitle>
            <DialogContent dividers>
                <EstablishmentHoursForm
                    mode="edit"
                    onClose={handleClose}
                    onEdit={() => void 0}
                    formik={formik}
                    loading={establishmentHoursLoading}
                />
            </DialogContent>
        </Dialog>
    )
}
