export enum BackendAppointmentStatus {
  SCHEDULED = 'SCHEDULED',
  COMPLETED = 'COMPLETED', 
  CANCELED = 'CANCELED'
}

export enum BackendAppointmentType {
  MEDICAL_VISIT = 'MEDICAL_VISIT',
  MEDICAL_PREFECTURE = 'MEDICAL_PREFECTURE',
  ANNUAL_INTERVIEW = 'ANNUAL_INTERVIEW',
  TRAINING = 'TRAINING',
  OTHER = 'OTHER'
}

export interface IAppointmentType {
  id?: string;
  appointmentType: BackendAppointmentType;
  appointmentStatus: BackendAppointmentStatus;
  medicalCenter: string | null;
  doctorName: string | null;
  appointmentDate: string; 
  timeSlot: string | null; 
  additionalDetails: string | null;
  driverId: number | null;
  driverFirstName: string | null;
  driverLastName: string | null;
  driverFullName: string | null;
  driverEmail: string | null;
  driverPhone: string | null;
  isActive: boolean;
  createdDate: string;
  lastModifiedDate: string;
}

export interface AppointmentSearchParams {
  id?: string;
  appointmentType?: BackendAppointmentType;
  appointmentStatus?: BackendAppointmentStatus;
  medicalCenter?: string;
  doctorName?: string;
  driverId?: number;
  driverName?: string;
  appointmentDate?: string;
  startDate?: string;
  endDate?: string;
  timeSlot?: string; 
  additionalDetails?: string;
  createdStartDate?: string;
  createdEndDate?: string;
  page?: number;
  size?: number;
  sortBy?: string;
  sortDirection?: 'asc' | 'desc';
}

export interface AppointmentPageResponse {
  content: IAppointmentType[];
  totalElements: number;
  totalPages: number;
  size: number;
  number: number;
  first: boolean;
  last: boolean;
  numberOfElements: number;
  empty: boolean;
}



export interface AppointmentStatusOption {
  value: BackendAppointmentStatus;
  label: string;
  color?: string;
  backgroundColor?: string;
}

export interface AppointmentTypeOption {
  value: BackendAppointmentType;
  label: string;
}

export interface IMedicalCenter {
  id: string;
  name: string;
  address: string;
}

export interface IDoctor {
  id: string;
  name: string;
  medicalCenterId: string;
}

export interface IMedicalReport {
  id: string;
  appointmentId: string;
  url: string;
  isValidated: boolean;
  validatedBy?: string;
  validatedAt?: string;
  comments?: string;
}

export type AppointmentStatus = 'prevu' | 'termine' | 'annule';

export type AppointmentType =
  | 'Visite Médicale (médecine du travail)'
  | 'Visite Médicale (Préfectorale)'
  | 'Entretien Annuel'
  | 'Formation'
  | 'Autre';

export interface IAppointment {
  id: string;
  driverId: string;
  driverName: string;
  type: AppointmentType;
  otherType: string;
  doctorId?: string;
  doctorName: string | null;
  medicalCenterId?: string;
  medicalCenterName: string | null;
  date: string;            
  time: string;            
  status: AppointmentStatus;
  comments: string | null;
  createdAt: string;
  updatedAt: string;
  driverFullName?: string | null;
  appointmentType?: BackendAppointmentType;
  medicalCenter?: string | null;
  appointmentStatus?: BackendAppointmentStatus;
  appointmentDate?: string;
  timeSlot?: string | null;
}

export const mapBackendStatusToFront: Record<BackendAppointmentStatus, AppointmentStatus> = {
  [BackendAppointmentStatus.SCHEDULED]: 'prevu',
  [BackendAppointmentStatus.COMPLETED]: 'termine',
  [BackendAppointmentStatus.CANCELED]:  'annule',
};

export const mapBackendTypeToFront: Record<BackendAppointmentType, AppointmentType> = {
  [BackendAppointmentType.MEDICAL_VISIT]:      'Visite Médicale (médecine du travail)',
  [BackendAppointmentType.MEDICAL_PREFECTURE]: 'Visite Médicale (Préfectorale)',
  [BackendAppointmentType.ANNUAL_INTERVIEW]:   'Entretien Annuel',
  [BackendAppointmentType.TRAINING]:           'Formation',
  [BackendAppointmentType.OTHER]:              'Autre',
};

// MODIFIÉ : Type pour les événements du calendrier (correspond au vrai CalendarEventDTO backend)
export interface CalendarEventDTO {
  id: number;
  appointmentType: BackendAppointmentType;
  appointmentStatus: BackendAppointmentStatus;
  medicalCenter: string | null;
  doctorName: string | null;
  appointmentDate: string; // LocalDate format ISO (YYYY-MM-DD)
  timeSlot: string | null; // LocalTime format ISO (HH:mm:ss)
  additionalDetails?: string | null;
  driverId: number | null;
  
  // OPTIONNELS : Ces champs peuvent ne pas être présents dans la réponse du calendrier
  driverFirstName?: string | null;
  driverLastName?: string | null;
  driverFullName?: string | null;
  driverEmail?: string | null;
  driverPhone?: string | null;
  isActive?: boolean;
  createdDate?: string;
  lastModifiedDate?: string;
}
