// Employee Status enum
export enum EmployeeStatus {
  EMPLOYEE = 'EMPLOYEE',
  SORTI = 'SORTI',
  ARCHIVE = 'ARCHIVE',
}

// Employee Type enum  
export enum EmployeeType {
  DRIVER = 'DRIVER',
  SEDENTARY = 'SEDENTARY',
}

// Base Employee interface
export interface IEmployee {
  id: string;
  employeeNumber: string;
  civility: string;
  lastName: string;
  firstName: string;
  birthDate: string;
  birthCity?: string;
  nationality: string;
  email: string;
  phone: string;
  address: string;
  postalCode: string;
  city: string;
  licenseDuration?: string;
  licenseNumber?: string;
  socialSecurityNumber?: string;
  status: EmployeeStatus;
  contractStartDate?: string;
  contractSignatureDate?: string;
  hourlyRate?: string;
  department: string;
  fuelType?: string;
  type: EmployeeType;
  
  // Document paths (stored in database)
  profilePicturePath?: string;
  healthCardPath?: string;
  bankDetailsPath?: string;
  licensePath?: string;
  idCardPath?: string;
  
  // Document URLs (generated presigned URLs)
  profilePictureUrl?: string;
  healthCardUrl?: string;
  bankDetailsUrl?: string;
  licenseUrl?: string;
  idCardUrl?: string;
  
  // Timestamps
  createdDate: string;
  lastModifiedDate: string;
  
  // Computed fields
  fullName: string;
}

// Create Employee Request
export interface CreateEmployeeRequest {
  // Personal information
  civility: string;
  lastName: string;
  firstName: string;
  birthDate: Date;
  birthCity: string;
  nationality: string;
  email: string;
  phone: string;
  address: string;
  postalCode: string;
  city: string;
  
  // Driving information
  licenseDuration?: string;
  licenseNumber?: string;
  socialSecurityNumber: string;
  
  // Professional information
  contractStartDate: Date;
  contractSignatureDate?: Date;
  hourlyRate: string;
  department: string;
  fuelType?: string;
  type: EmployeeType;
  
  // Document files for upload
  profilePictureFile?: File;
  healthCardFile?: File;
  bankDetailsFile?: File;
  licenseFile?: File;
  idCardFile?: File;
}

// Update Employee Request
export interface UpdateEmployeeRequest extends CreateEmployeeRequest {
  status?: EmployeeStatus;
  
  // Existing document paths (for keeping existing files)
  existingProfilePicturePath?: string;
  existingHealthCardPath?: string;
  existingBankDetailsPath?: string;
  existingLicensePath?: string;
  existingIdCardPath?: string;
}

// Page Response for pagination
export interface PageResponse<T> {
  content: T[];
  totalElements: number;
  totalPages: number;
  size: number;
  number: number;
  first: boolean;
  last: boolean;
  numberOfElements: number;
}

// Employee filters for search
export interface EmployeeFilters {
  lastName?: string;
  firstName?: string;
  email?: string;
  phone?: string;
  employeeNumber?: string;
  department?: string;
  status?: EmployeeStatus;
  type?: EmployeeType;
  registrationDateFrom?: string;
  registrationDateTo?: string;
  searchTerm?: string;
}

// Employee sort options
export interface EmployeeSort {
  sortBy: string;
  sortDirection: 'asc' | 'desc';
}

// Employee pagination
export interface EmployeePagination {
  page: number;
  size: number;
}

// Combined search params
export interface EmployeeSearchParams extends EmployeeFilters, EmployeeSort, EmployeePagination {}

