import { InterventionType, ServiceProvider } from "./service-provider";

export enum IncidentStatus {
    IN_PROGRESS = 'IN_PROGRESS',
    RESOLVED = 'RESOLVED',
    AWAITING_PARTS = 'AWAITING_PARTS',
    UNDER_REPAIR = 'UNDER_REPAIR'
}

export const IncidentStatusLabels: Record<IncidentStatus, string> = {
    [IncidentStatus.IN_PROGRESS]: 'En cours',
    [IncidentStatus.RESOLVED]: 'Résolu',
    [IncidentStatus.AWAITING_PARTS]: 'En attente de pièces',
    [IncidentStatus.UNDER_REPAIR]: 'En réparation'
}

export const IncidentStatusOptions = [
    { value: IncidentStatus.IN_PROGRESS, label: IncidentStatusLabels[IncidentStatus.IN_PROGRESS] },
    { value: IncidentStatus.RESOLVED, label: IncidentStatusLabels[IncidentStatus.RESOLVED] },
    { value: IncidentStatus.AWAITING_PARTS, label: IncidentStatusLabels[IncidentStatus.AWAITING_PARTS] },
    { value: IncidentStatus.UNDER_REPAIR, label: IncidentStatusLabels[IncidentStatus.UNDER_REPAIR] }
]

export const getIncidentStatusColor = (status: IncidentStatus) => {
    switch (status) {
        case IncidentStatus.IN_PROGRESS:
            return '#F9B90B';
        case IncidentStatus.RESOLVED:
            return '#22c55e';
        case IncidentStatus.AWAITING_PARTS:
            return '#FF9800';
        case IncidentStatus.UNDER_REPAIR:
            return '#81919A';
        default:
            return '#cccccc';
    }
}

export enum DeclarationType {
    AXA = 'AXA',
    ADANEV = 'ADANEV'
}

export const DeclarationTypeLabels = {
    [DeclarationType.AXA]: 'AXA',
    [DeclarationType.ADANEV]: 'Adanev'
}

export const DeclarationTypeOptions = [
    { value: DeclarationType.AXA, label: DeclarationTypeLabels[DeclarationType.AXA] },
    { value: DeclarationType.ADANEV, label: DeclarationTypeLabels[DeclarationType.ADANEV] }
]

export type VehicleIncident = {
    id: string;
    departmentId: string;
    departmentName: string;
    incidentDate: string;
    vehicleId: string;
    vehicleRegistrationPlate: string;
    driverId: string;
    driverName: string;
    damages: string;
    circumstances: string;
    responsibility: string;
    declarationType: DeclarationType;
    serviceProvider: ServiceProvider
    cost: number;
    status: IncidentStatus;
    active: boolean;
};

export type VehicleIncidentRequest = {
    id: string;
    incidentDate: string;
    vehicleId: string;
    departmentId: string;
    driverId: string;
    damages: string;
    circumstances: string;
    responsibility: string;
    declarationType: DeclarationType;
    serviceProviderId?: string;
    interventionId?: string;
    interventionType?: InterventionType;
    cost: number;
};

export type VehicleIncidentPageFilterParams = {
    id?: number;
    departmentName?: string;
    incidentDate?: string;
    vehicleRegistrationPlate?: string;
    driverName?: string;
    declarationType?: DeclarationType;
    status?: IncidentStatus;
    page?: number;
    size?: number;
    orderBy?: string;
    orderDir?: 'asc' | 'desc';
};

export const mapVehicleIncidentToRequest = (vehicle: VehicleIncident): VehicleIncidentRequest => {
    return {
        id: vehicle.id,
        incidentDate: vehicle.incidentDate,
        vehicleId: vehicle.vehicleId,
        departmentId: vehicle.departmentId,
        driverId: vehicle.driverId,
        damages: vehicle.damages,
        circumstances: vehicle.circumstances,
        responsibility: vehicle.responsibility,
        declarationType: vehicle.declarationType,
        serviceProviderId: vehicle.serviceProvider?.id,
        interventionId: vehicle.serviceProvider.intervention.id,
        interventionType: vehicle.serviceProvider.intervention.interventionType,
        cost: vehicle.cost
    };
};