import { FuelType } from "./fuelType";
import { VehicleBrandModel } from "./vehicle-brand-model";

export enum VehicleStatus {
    ACTIVE = 'ACTIVE',
    INACTIVE = 'INACTIVE',
    UNDER_REPAIR = 'UNDER_REPAIR'
}

export const VehicleStatusLabels = {
    [VehicleStatus.ACTIVE]: 'Actif',
    [VehicleStatus.INACTIVE]: 'Inactif',
    [VehicleStatus.UNDER_REPAIR]: 'En réparation'
}

export const VehicleStatusOptions = [
    { value: VehicleStatus.ACTIVE, label: VehicleStatusLabels[VehicleStatus.ACTIVE] },
    { value: VehicleStatus.INACTIVE, label: VehicleStatusLabels[VehicleStatus.INACTIVE] },
    { value: VehicleStatus.UNDER_REPAIR, label: VehicleStatusLabels[VehicleStatus.UNDER_REPAIR] }
]

export const getVehicleStatusColor = (status: VehicleStatus) => {
    switch (status) {
        case VehicleStatus.ACTIVE:
            return '#22c55e';
        case VehicleStatus.INACTIVE:
            return '#FF3434';
        case VehicleStatus.UNDER_REPAIR:
            return '#81919A';
        default:
            return '#F9B90B';
    }
}

export type VehicleDropdown = {
    id: string;
    vehicleIdentificationNumber: string;
    registrationPlate: string;
};

export type VehicleListResponse = {
    id: string;
    vehicleIdentificationNumber: string;
    status: VehicleStatus;
    registrationPlate: string;
    driverName: string;
    brandName: string;
    modelName: string;
    departmentName: string;
    fuelTypeName: string;
    vehicleTypeName: string;
    active: boolean;
};

export type VehicleRequest = {
    id: string;
    vehicleIdentificationNumber: string;
    status: VehicleStatus;
    registrationPlate: string;
    registrationDate: string;
    driverId: string;
    brandName: string;
    modelName: string;
    vehicleTypeId: string;
    vehicleBrandModelId: string;
    departmentId: string;
    fuelTypeId: string;
    seatCount: number;
    wheelchairSeatCount: number;
    accidented: boolean;
    observation?: string;
    locationContractCode?: string;
    orderDate?: string;
    deliveryDate?: string;
    locationPeriod: number;
    proprietor?: string;
    locationTariff: number;
    sectorId?: string;
    licenceNumber?: string;
    geolocationCode?: string;
    active: boolean;
};

export type Vehicle = {
    id: string;
    vehicleIdentificationNumber: string;
    status: VehicleStatus;
    registrationPlate: string;
    registrationDate: string;
    driverId: string;
    vehicleBrandModel: VehicleBrandModel;
    departmentId: string;
    fuelType: FuelType;
    seatCount: number;
    wheelchairSeatCount: number;
    accidented: boolean;
    observation?: string;
    locationContractCode?: string;
    orderDate?: string;
    deliveryDate?: string;
    locationPeriod: number;
    proprietor?: string;
    locationTariff: number;
    sectorId?: string;
    licenceNumber?: string;
    geolocationCode?: string;
    active: boolean;
};

export type VehiclePageFilterParams = {
    id?: string;
    brandName?: string;
    modelName?: string;
    status?: VehicleStatus;
    registrationPlate?: string;
    vehicleTypeName?: string;
    fuelTypeName?: string;
    driverName?: string;
    departmentName?: string;
    page?: number;
    size?: number;
    orderBy?: string;
    orderDir?: 'asc' | 'desc';
};

export const mapVehicleToRequest = (vehicle: Vehicle): VehicleRequest => {
    return {
        id: vehicle.id,
        vehicleIdentificationNumber: vehicle.vehicleIdentificationNumber,
        status: vehicle.status,
        registrationPlate: vehicle.registrationPlate,
        registrationDate: vehicle.registrationDate,
        driverId: vehicle.driverId,
        brandName: vehicle.vehicleBrandModel.brandName,
        modelName: vehicle.vehicleBrandModel.modelName,
        vehicleTypeId: vehicle.vehicleBrandModel.vehicleType.id,
        vehicleBrandModelId: vehicle.vehicleBrandModel.id,
        departmentId: vehicle.departmentId,
        fuelTypeId: vehicle.fuelType.id,
        seatCount: vehicle.seatCount,
        wheelchairSeatCount: vehicle.wheelchairSeatCount,
        accidented: vehicle.accidented,
        observation: vehicle.observation,
        locationContractCode: vehicle.locationContractCode,
        orderDate: vehicle.orderDate,
        deliveryDate: vehicle.deliveryDate,
        locationPeriod: vehicle.locationPeriod,
        proprietor: vehicle.proprietor,
        locationTariff: vehicle.locationTariff,
        sectorId: vehicle.sectorId,
        licenceNumber: vehicle.licenceNumber,
        geolocationCode: vehicle.geolocationCode,
        active: vehicle.active,
    };
};