import { IRHAbsence } from '../types/absence-rh';
import { TableColumn } from '../types/common';
import { DriverAbsenceStatus } from '../types/driver-absence';

// Types d'absence
export const _TYPES_ABSENCE = ["Maladie", "Exceptionnelle", "Autorisée", "Formation", "Congé payé"];

// Chauffeurs
export const _CHAUFFEURS = ["Jean Dupont", "Marie Durand", "Pierre Martin", "Sophie Lefebvre", "Ahmed Benali"];

// Modes de réception
export const _MODES_RECEPTION = ["Portail", "Application mobile", "Email", "Téléphone", "Courrier"];

// Départements
export const _DEPARTEMENTS = ["Transport", "Logistique", "Administration", "Commercial", "Maintenance"];

const TOTAL_ABSENCES = 30;

// Fonction pour générer des données d'absence aléatoires
const generateRHAbsences = (etats: string[]) => {
  const absences: IRHAbsence[] = [];
  
  // Fonction pour générer une date aléatoire entre deux dates
  const getRandomDate = (start: Date, end: Date) => {
    return new Date(start.getTime() + Math.random() * (end.getTime() - start.getTime()));
  };
  
  // Fonction pour formater une date en YYYY-MM-DD
  const formatDate = (date: Date) => {
    return date.toISOString().split("T")[0];
  };
  
  // Types possibles pour le personnel
  const types: Array<'chauffeur' | 'sédentaire'> = ['chauffeur', 'sédentaire'];
  
  for (let i = 0; i < TOTAL_ABSENCES; i++) {
    // Générer une date de début aléatoire dans les 3 derniers mois
    const today = new Date();
    const threeMonthsAgo = new Date();
    threeMonthsAgo.setMonth(today.getMonth() - 3);
    
    const startDate = getRandomDate(threeMonthsAgo, today);
    
    // Générer une date de fin entre 1 et 14 jours après la date de début
    const maxEndDate = new Date(startDate);
    maxEndDate.setDate(startDate.getDate() + 14);
    
    const endDate = getRandomDate(startDate, maxEndDate);
    const statusList = Object.values(DriverAbsenceStatus);

    absences.push({
      id: (i + 1).toString(),
      numeroEnregistrement: `ABS-${(i + 1).toString().padStart(5, "0")}`,
      nomChauffeur: _CHAUFFEURS[i % _CHAUFFEURS.length],
      type: types[i % types.length],
      typeAbsence: _TYPES_ABSENCE[i % _TYPES_ABSENCE.length],
      dateDebut: formatDate(startDate),
      dateFin: formatDate(endDate),
      statut: statusList[Math.floor(Math.random() * statusList.length)],
      justificatif: i % 2 === 0 ? "Certificat médical" : "Attestation",
      affecteTrajet: i % 3 === 0 ? "oui" : "non",
      modeReception: _MODES_RECEPTION[i % _MODES_RECEPTION.length],
      nbrHeures: Math.floor(Math.random() * 40) + 1,
      validerPar: (statusList[i % statusList.length] === DriverAbsenceStatus.EnCours || statusList[i % statusList.length] === DriverAbsenceStatus.NonTraite) 
                  ? "" : _CHAUFFEURS[(i + 2) % _CHAUFFEURS.length],
      autorisePar: i % 2 === 0 ? _CHAUFFEURS[(i + 1) % _CHAUFFEURS.length] : "",
      departement: _DEPARTEMENTS[i % _DEPARTEMENTS.length]
    });
  }
  return absences;
};

const etatsTraites = [DriverAbsenceStatus.Valide, DriverAbsenceStatus.Rejete];
const etatsRecus = [DriverAbsenceStatus.EnCours, DriverAbsenceStatus.NonTraite];

// Données d'absence générées
export const _absencesTraitees = generateRHAbsences(etatsTraites);
export const _absencesRecues = generateRHAbsences(etatsRecus);

// Fonction pour obtenir le libellé d'un statut
export const getRHAbsenceStatut = (status: string): string => {
  return status || "Inconnu";
};

// Fonction pour obtenir le libellé d'un état
export const getAbsenceEtat = (status: string): string => {
  return Object.values(DriverAbsenceStatus).includes(status as DriverAbsenceStatus) ? status : "Inconnu";
};

// Colonnes par défaut pour le tableau d'absences
export const DEFAULT_ABSENCE_TABLE_HEAD: TableColumn[] = [
  { id: "numeroEnregistrement", label: "N° d'enregistrement", type: "text", align: "center" },
  { id: "nomChauffeur", label: "Nom et prénom", type: "text", align: "left" },
  { id: "typeAbsence", label: "Type d'absence", type: "text", align: "left" },
  { id: "dateDebut", label: "Date de début", type: "date", align: "center" },
  { id: "dateFin", label: "Date de fin", type: "date", align: "center" },
  { id: "nbrHeures", label: "Nbr des heures", type: "number", align: "center" },
  { id: "autorisePar", label: "Autorise par", type: "text", align: "left" },
  { id: "validerPar", label: "Valider par", type: "text", align: "left" },
  { id: "departement", label: "Département", type: "text", align: "left" },
  { id: "statut", label: "Statut de la demande", type: "text", align: "left" },
  { id: "modeReception", label: "Mode de réception", type: "text", align: "left" },
  { id: "type", label: "Type", type: "text", align: "left" },
  { id: "affecteTrajet", label: "Affecté au trajet", type: "text", align: "left" },
  { id: "justificatif", label: "Justificatif", type: "text", align: "center" },
];

// Colonnes pour le tableau des absences non traitées (sans le champ "Valider par")
export const DEFAULT_ABSENCE_NONTREATED_TABLE_HEAD: TableColumn[] = [
  { id: "numeroEnregistrement", label: "N° d'enregistrement", type: "text", align: "center" },
  { id: "nomChauffeur", label: "Nom et prénom", type: "text", align: "left" },
  { id: "typeAbsence", label: "Type d'absence", type: "text", align: "left" },
  { id: "dateDebut", label: "Date de début", type: "date", align: "center" },
  { id: "dateFin", label: "Date de fin", type: "date", align: "center" },
  { id: "nbrHeures", label: "Nbr des heures", type: "number", align: "center" },
  { id: "autorisePar", label: "Autorise par", type: "text", align: "left" },
  { id: "departement", label: "Département", type: "text", align: "left" },
  { id: "statut", label: "Statut de la demande", type: "text", align: "left" },
  { id: "modeReception", label: "Mode de réception", type: "text", align: "left" },
  { id: "type", label: "Type", type: "text", align: "left" },
  { id: "affecteTrajet", label: "Affecté au trajet", type: "text", align: "left" },
  { id: "justificatif", label: "Justificatif", type: "text", align: "center" },
  { id: '', label: 'Action', type: 'text', align: 'center' },
];

// Données initiales pour un nouveau formulaire d'absence
export const INITIAL_ABSENCE_DATA: IRHAbsence = {
  id: "",
  numeroEnregistrement: "",
  nomChauffeur: "",
  type: 'chauffeur',
  typeAbsence: _TYPES_ABSENCE[0],
  dateDebut: "",
  dateFin: "",
  statut: DriverAbsenceStatus.EnCours,
  justificatif: "",
  affecteTrajet: 'non',
  modeReception: _MODES_RECEPTION[0],
  nbrHeures: 0,
  validerPar: "",
  autorisePar: "",
  departement: ""
}; 
