import { create } from 'zustand';
import axiosInstance from '@/utils/axios';
import { IAbsence, EtatDemandeRecue, EtatDemandeTraitee, ReceivingMethod } from '@/shared/types/absence';
import { endpoints } from '../endpoints/absence-service';
import { mapSensToConcernedTrip } from '@/shared/types/absence';

interface AbsencePageFilterParams {
  page?: number;
  size?: number;
  sortBy?: string;
  sortDirection?: 'asc' | 'desc';
  state?: string[];
  passengerId?: number;
  receivingMethod?: string;
  establishment?: string;
  departmentName?: string;
  lastName?: string;
  firstName?: string;
  startDateFrom?: string;
  startDateTo?: string;
}

interface PassengerResponse {
  content: Array<{
    id: number;
    firstName: string;
    lastName: string;
    establishmentName: string;
    departmentName: string;
  }>;
}

interface ExportFilters {
  firstName?: string;
  lastName?: string;
  state?: string;
  startDate?: string;
  endDate?: string;
  reportedBy?: string;
  userId?: string;
}

// Map frontend state to backend state
const mapEtatToBackendState = (etat: string): string => {
  switch (etat) {
    case 'En cours':
      return 'In_progress';
    case 'Non traité':
      return 'Unprocessed';
    case 'Annulé':
      return 'Cancelled';
    case 'Traité':
      return 'Processed';
    default:
      return 'Unprocessed';
  }
};

type AbsenceStore = {
  loading: boolean;
  error: string | null;
  unprocessedAbsences: IAbsence[];
  processedAbsences: IAbsence[];
  passengers: { id: number; value: string; label: string; establishmentName: string; departmentName: string }[];
  fetchUnprocessedAbsences: (params?: AbsencePageFilterParams) => Promise<void>;
  fetchProcessedAbsences: (params?: AbsencePageFilterParams) => Promise<void>;
  fetchPassengers: () => Promise<void>;
  absences: IAbsence[];
  fetchAbsences: () => Promise<void>;
  absenceById: IAbsence | null;
  getAbsenceById: (id: string) => Promise<IAbsence>;
  updateAbsence: (id: number, absence: Partial<IAbsence>) => Promise<void>;
  createAbsence: (createData: any) => Promise<IAbsence>;
  startExportAbsences: (params?: any) => Promise<any>;
};

export const useAbsenceStore = create<AbsenceStore>((set, get) => ({
  loading: false,
  error: null,
  unprocessedAbsences: [],
  processedAbsences: [],
  passengers: [],
  fetchPassengers: async () => {
    try {
      const response = await axiosInstance.get<PassengerResponse>(endpoints.User.searchPassengers);
      set(state => ({
        ...state,
        passengers: response.data.content.map(passenger => ({
          id: passenger.id,
          value: passenger.id.toString(),
          label: `${passenger.firstName} ${passenger.lastName}`,
          establishmentName: passenger.establishmentName,
          departmentName: passenger.departmentName
        }))
      }));
    } catch (error) {
      console.error('Failed to fetch passengers:', error);
    }
  },
  fetchUnprocessedAbsences: async (params?: AbsencePageFilterParams) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.get(endpoints.Absence.getAbsences, {
        params: {
          ...params,
          state: ['Unprocessed', 'In_progress']
        },
        paramsSerializer: {
          indexes: null // This will format arrays as repeated params instead of using brackets
        }
      });
      set({
        unprocessedAbsences: response.data.content,
        loading: false,
      });
    } catch (error: any) {
      set({
        error: error.message || 'Failed to fetch unprocessed absences',
        loading: false,
      });
    }
  },
  fetchProcessedAbsences: async (params?: AbsencePageFilterParams) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.get(endpoints.Absence.getAbsences, {
        params: {
          ...params,
          state: ['Processed', 'Cancelled']
        },
        paramsSerializer: {
          indexes: null // This will format arrays as repeated params instead of using brackets
        }
      });
      set({
        processedAbsences: response.data.content,
        loading: false,
      });
    } catch (error: any) {
      set({
        error: error.message || 'Failed to fetch processed absences',
        loading: false,
      });
    }
  },
  absences: [],
  fetchAbsences: async () => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.get(endpoints.Absence.getAbsences);
      set({
        absences: response.data.content,
        loading: false,
      });
    } catch (error: any) {
      set({
        error: error.message || 'Failed to fetch absences',
        loading: false,
      });
    }
  },
  absenceById: null,
  getAbsenceById: async (id: string) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.get(endpoints.Absence.getAbsenceById(id));
      set({
        absenceById: response.data,
        loading: false,
      });
      return response.data;
    } catch (error: any) {
      set({
        error: error.message || 'Failed to get absence by id',
        loading: false,
      });
    }
  },
  updateAbsence: async (id: number, absence: Partial<IAbsence>) => {
    set({ loading: true, error: null });
    try {
      await axiosInstance.put(endpoints.Absence.updateAbsence(id), absence);
      
      await Promise.all([
        get().fetchUnprocessedAbsences(),
        get().fetchProcessedAbsences(),
      ]);

      set({ loading: false });
    } catch (error: any) {
      set({
        error: error.message || 'Failed to update absence',
        loading: false,
      });
      throw error;
    }
  },
  createAbsence: async (createData: any) => {
    set({ loading: true, error: null });
    try {
      
      const response = await axiosInstance.post(
        endpoints.Absence.createAbsence,
        createData
      );      await Promise.all([
        get().fetchUnprocessedAbsences(),
        get().fetchProcessedAbsences(),
      ]);

      set({ loading: false });
      return response.data;
    } catch (error: any) {
      console.error('Error creating absence:', error); // Debug log
      set({
        error: error.message || 'Failed to create absence',
        loading: false,
      });
      throw error;
    }
  },
  startExportAbsences: async (params?: any) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.post(endpoints.Absence.exportAbsence, { params });
      set({ loading: false });
      return response.data;
    } catch (error: any) {
      set({
        error: error.message || 'Failed to start export absences',
        loading: false,
      });
      throw error;
    }
  },
}));