'use client';

import React, { useState, useEffect } from 'react';
import { Typography, Grid, Box, Paper, Divider } from '@mui/material';
import { LocalizationProvider } from '@mui/x-date-pickers/LocalizationProvider';
import { AdapterDateFns } from '@mui/x-date-pickers/AdapterDateFns';
import { fr } from 'date-fns/locale';
import FormField from '@/shared/components/form/form-field';
import { FormFieldType, ModeType } from '@/shared/types/common';
import FormContainer from '@/layouts/form/form-container';
import EditExportButtons from '@/shared/components/form/edit-export-buttons';
import ActionButtons from '@/shared/components/form/buttons-action';
import { rentalFormFields } from '../utils/form-fields-rental';
import { IRental } from '@/shared/types/rental';
import FontAwesome from '@/shared/components/fontawesome';
import { faCar, faCalendarDays, faTag, faBuilding, faInfoCircle, faIdCard } from '@fortawesome/free-solid-svg-icons';

interface VehicleRentalFormProps {
  rental: IRental;
  mode: ModeType;
  onSave?: (updatedRental: IRental) => void;
  onEdit?: (updatedRental: IRental) => void;
  onClose: () => void;
  isDrawerElement?: boolean;
  hasBorder?: boolean;
}

const VehicleRentalForm: React.FC<VehicleRentalFormProps> = ({
  rental,
  mode,
  onSave,
  onClose,
  onEdit,
  isDrawerElement,
  hasBorder
}) => {
  const [formData, setFormData] = useState<IRental>(rental);

  useEffect(() => {
    setFormData(rental);
  }, [rental]);

  const handleSave = () => {
    if (onSave) {
      onSave(formData);
    }
    onClose();
  };

  const isReadOnly = mode === 'view';

  const renderField = (field: FormFieldType<IRental>) => (
    <FormField
      field={field}
      value={formData[field.name]}
      onChange={(name, value) => setFormData((prev) => ({ ...prev, [name]: value }))}
      isReadOnly={isReadOnly}
      isDrawerElement={isDrawerElement}
    />
  );

  const renderHeader = () => (
    <Box sx={{ position: 'relative', mb: 3 }}>
      <Typography 
        variant="h4" 
        sx={{ 
          color: 'primary.main', 
          fontWeight: 600 
        }}
      >
        Fiche Location
      </Typography>

      {isReadOnly && onEdit && (
        <Box 
          sx={{ 
            position: 'absolute', 
            top: 16, 
            right: 16 
          }}
        >
          <EditExportButtons
            onEdit={() => onEdit(formData)}
            onExport={() => void 0}
            tooltipTitle="la location"
          />
        </Box>
      )}
    </Box>
  );

  if (!formData && isReadOnly) {
    return (
      <Box sx={{ p: isDrawerElement ? 2 : 3, width: '100%' }}>
        <Typography>
          Aucune donnée disponible
        </Typography>
      </Box>
    );
  }

  return (
    <LocalizationProvider dateAdapter={AdapterDateFns} adapterLocale={fr}>
      <Box sx={{ p: isDrawerElement ? 2 : 3, width: '100%' }}>
        {renderHeader()}

        <form>
          <Paper elevation={0} sx={{ p: 2, mb: 3, borderRadius: 2, border: '1px solid', borderColor: 'divider' }}>
            <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
              <FontAwesome icon={faCar} color="primary.main" width={16} sx={{ mr: 1 }} />
              <Typography variant="subtitle1" color="primary.main" fontWeight={600}>
                Informations du véhicule
              </Typography>
            </Box>
            <Grid container spacing={2}>
              <Grid item xs={12} md={4}>
                {renderField(rentalFormFields.find(f => f.name === 'Immatriculation')!)}
              </Grid>
              <Grid item xs={12} md={4}>
                {renderField(rentalFormFields.find(f => f.name === 'vehicleType')!)}
              </Grid>
              <Grid item xs={12} md={4}>
                {renderField(rentalFormFields.find(f => f.name === 'supplier')!)}
              </Grid>
              {rentalFormFields.find(f => f.name === 'immatriculation') && (
                <Grid item xs={12} md={4}>
                  {renderField(rentalFormFields.find(f => f.name === 'immatriculation')!)}
                </Grid>
              )}
            </Grid>
          </Paper>
          
          <Paper elevation={0} sx={{ p: 2, mb: 3, borderRadius: 2, border: '1px solid', borderColor: 'divider' }}>
            <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
              <FontAwesome icon={faCalendarDays} color="primary.main" width={16} sx={{ mr: 1 }} />
              <Typography variant="subtitle1" color="primary.main" fontWeight={600}>
                Période de location
              </Typography>
            </Box>
            
            <Box sx={{ mb: 2 }}>
              <Typography variant="subtitle2" color="text.secondary" sx={{ mb: 1 }}>
                Début
              </Typography>
              <Grid container spacing={2}>
                <Grid item xs={12} md={6}>
                  {renderField(rentalFormFields.find(f => f.name === 'startDate')!)}
                </Grid>
                <Grid item xs={12} md={6}>
                  {renderField(rentalFormFields.find(f => f.name === 'startTime')!)}
                </Grid>
              </Grid>
            </Box>
            
            <Box>
              <Typography variant="subtitle2" color="text.secondary" sx={{ mb: 1 }}>
                Fin
              </Typography>
              <Grid container spacing={2}>
                <Grid item xs={12} md={6}>
                  {renderField(rentalFormFields.find(f => f.name === 'endDate') || rentalFormFields.find(f => f.name === 'startDate')!)}
                </Grid>
                <Grid item xs={12} md={6}>
                  {renderField(rentalFormFields.find(f => f.name === 'endTime')!)}
                </Grid>
              </Grid>
            </Box>
          </Paper>
          
          <Paper elevation={0} sx={{ p: 2, mb: 3, borderRadius: 2, border: '1px solid', borderColor: 'divider' }}>
            <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
              <FontAwesome icon={faTag} color="primary.main" width={16} sx={{ mr: 1 }} />
              <Typography variant="subtitle1" color="primary.main" fontWeight={600}>
                Status et observations
              </Typography>
            </Box>
            <Grid container spacing={2}>
              <Grid item xs={12} md={4}>
                {renderField(rentalFormFields.find(f => f.name === 'status')!)}
              </Grid>
              <Grid item xs={12} md={8}>
                {renderField(rentalFormFields.find(f => f.name === 'notes')!)}
              </Grid>
            </Grid>
          </Paper>

          {mode !== 'view' ? (
            <Box sx={{ display: 'flex', justifyContent: 'flex-end', mt: 3 }}>
              <ActionButtons onSave={handleSave} onCancel={onClose} mode={mode} />
            </Box>
          ) : <></>}
        </form>
      </Box>
    </LocalizationProvider>
  );
};

export default VehicleRentalForm; 