import FontAwesome from '@/shared/components/fontawesome';
import { pxToRem } from '@/shared/theme/typography';
import { 
  faCar, 
  faWrench, 
  faCalendarCheck, 
  faInfoCircle,
  faUserTie 
} from '@fortawesome/free-solid-svg-icons';
import { Box, Popover, Typography } from '@mui/material';
import { getBgColor } from '../utils/schedule-utils';
import { InterventionType } from '@/shared/_mock/_planningEvents';

interface PopoverData {
  type: InterventionType;
  title: string;
  schedule: string;
  details: string;
  technician: string;
  vehicle: {
    immatriculation: string;
    modele: string;
    marque: string;
  };
}

interface VehiculeDetailsPopoverProps {
  anchorEl: HTMLElement | null;
  onClose: () => void;
  color: string;
  data: PopoverData;
}

export function VehiculeDetailsPopover({ anchorEl, onClose, data, color }: VehiculeDetailsPopoverProps) {
  const bgColor = getBgColor(color);
  
  if (!data) return null;

  return (
    <Popover
      open={Boolean(anchorEl)}
      anchorEl={anchorEl}
      onClose={onClose}
      disableRestoreFocus
      anchorOrigin={{
        vertical: 'center',
        horizontal: 'center',
      }}
      transformOrigin={{
        vertical: 'top',
        horizontal: 'center',
      }}
      sx={{
        pointerEvents: 'none',
        '& .MuiPopover-paper': {
          p: 1.5,
          mt: 3,
          borderRadius: 4,
          border: `2px solid ${color}`,
          bgcolor: bgColor,
          overflow: 'visible',
          boxShadow: 'none',
          '&::before': {
            content: '""',
            position: 'absolute',
            top: -10,
            left: '45%',
            width: '10%',
            aspectRatio: '1 / 1',
            backgroundColor: bgColor,
            transform: 'rotate(45deg)',
            borderRadius: 1,
            border: `2px solid ${color}`,
            borderBottom: 'none',
            borderRight: 'none',
            zIndex: 1,
          },
        },
      }}
    >
      <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1, minWidth: 200 }}>
        {/* Title/Type */}
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faWrench} color={color} width={13} />
          <Box>
            <Typography sx={{ fontSize: pxToRem(12), color: color, fontWeight: 600 }}>
              {data.title}
            </Typography>
            <Typography sx={{ fontSize: pxToRem(11), color: color }}>
              {`Intervention ${data.type === 'preventive' ? 'Préventive' : 'Corrective'}`}
            </Typography>
          </Box>
        </Box>

        {/* Vehicle Information */}
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faCar} color={color} width={13} />
          <Box>
            <Typography sx={{ fontSize: pxToRem(12), color: color }}>
              {`${data.vehicle.marque} ${data.vehicle.modele}`}
            </Typography>
            <Typography sx={{ fontSize: pxToRem(11), color: color }}>
              {data.vehicle.immatriculation}
            </Typography>
          </Box>
        </Box>

        {/* Schedule */}
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faCalendarCheck} color={color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: color }}>
            {data.schedule}
          </Typography>
        </Box>

        {/* Technician */}
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faUserTie} color={color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: color }}>
            {data.technician}
          </Typography>
        </Box>

        {/* Details */}
        {data.details && (
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
            <FontAwesome icon={faInfoCircle} color={color} width={13} />
            <Typography sx={{ fontSize: pxToRem(12), color: color }}>
              {data.details}
            </Typography>
          </Box>
        )}
      </Box>
    </Popover>
  );
} 