"use client"

import type React from "react"
import { useEffect, useRef, useState } from "react"
import { useFormik } from "formik"
import { Grid, Stack, Box, IconButton, Snackbar, Alert, InputAdornment, List, ListItem, ListItemIcon, ListItemText, Typography } from "@mui/material"
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome"
import { faPaperclip, faExclamationCircle } from "@fortawesome/free-solid-svg-icons"
import { type ICandidature, CandidatureStatut } from "@/shared/types/candidature"
import { type ModeType, TableType, type TableColumn } from "@/shared/types/common"
import FormField from "@/shared/components/form/form-field"
import { StyledTextField } from "@/shared/theme/css"
import { useResponsive } from "@/hooks"
import { getCandidatureFields } from "../../utils/form-fields-formulaire"
import FormContainer from "@/layouts/form/form-container"
import EditExportButtons from "@/shared/components/form/edit-export-buttons"
import CustomTooltip from "@/shared/components/tooltips/tooltip-custom"
import { AddButton } from "@/shared/components/table/styles"
import ActionButtons from "@/shared/components/form/buttons-action"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import SendEmailModal from "../send-email-modal"
import { createCandidatureSchema } from "../../utils/candidature-schema"

interface CandidatureFormProps {
  candidature: ICandidature
  mode: ModeType
  onSave: (candidature: ICandidature) => void
  onClose: (forceClose: boolean) => void;
  onEdit?: (candidature: ICandidature) => void
  tableHead: TableColumn[]
  updateTabContent: (tabId: string, newContent: ICandidature) => void
  tabId: string
}

export default function CandidatureDetailsForm({
  candidature,
  mode,
  onSave,
  onClose,
  onEdit,
  tableHead,
  updateTabContent,
  tabId,
  }: CandidatureFormProps) {
  const documentFileInputRef = useRef<HTMLInputElement>(null)
  const contratDocumentFileInputRef = useRef<HTMLInputElement>(null)
  const isBetweenSmAndLg = useResponsive("between", "sm", "lg")

  // State for email modal
  const [isEmailModalOpen, setIsEmailModalOpen] = useState(false)
  const [emailSubject, setEmailSubject] = useState("")
  const [emailMessage, setEmailMessage] = useState("")

  // State for notifications
  const [notification, setNotification] = useState({
    open: false,
    message: "",
    severity: "info" as "success" | "info" | "warning" | "error",
  })

  const isViewMode = mode === "view"
  const isEditMode = mode === "edit"
  const isNewMode = mode === "add"

  // Initialisation de Formik
  const formik = useFormik<ICandidature>({
    initialValues: candidature,
    validationSchema: createCandidatureSchema(candidature.type === "Candidature sédentaire"),
    enableReinitialize: true,
    onSubmit: (values) => {
      // Vérifier si le statut nécessite une redirection
      const statusRequiresRedirection =
        values.statut === CandidatureStatut.Retenu ||
        values.statut === CandidatureStatut.Refuse ||
        values.statut === CandidatureStatut.Blackliste ||
        values.statut === CandidatureStatut.Valide ||
        values.statut === CandidatureStatut.Absent

      // Pour le statut Valide, s'assurer que le contrat est défini
      if (values.statut === CandidatureStatut.Valide && !values.contrat) {
        values.contrat = "non signé"
      }

      // Sauvegarder la candidature
      onSave(values)

      // Afficher une notification en fonction du changement de statut
      if (values.statut === CandidatureStatut.Valide) {
        setNotification({
          open: true,
          message: "Candidature validée. Elle est maintenant disponible dans l'onglet Tableau",
          severity: "success",
        })
      } else if (values.statut === CandidatureStatut.Retenu) {
        setNotification({
          open: true,
          message: "Candidature retenue. Elle est maintenant disponible dans l'onglet Retenus",
          severity: "success",
        })
      } else if (values.statut === CandidatureStatut.Refuse || values.statut === CandidatureStatut.Blackliste) {
        setNotification({
          open: true,
          message:
            "Candidature refusée/blacklistée. Elle est maintenant disponible dans l'onglet Historique des candidatures",
          severity: "info",
        })
      } else if (values.statut === CandidatureStatut.Absent) {
        setNotification({
          open: true,
          message: "Candidature marquée comme absente. Elle est maintenant disponible dans l'onglet Absent RDV",
          severity: "info",
        })
      }
      // Fermer le formulaire si le statut nécessite une redirection
      if (statusRequiresRedirection) {
        onClose(true)
      }
    },
    
  })

  useEffect(() => {
    if (updateTabContent && tabId && mode !== "view") {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const isSedentaire = formik.values.type === "Candidature sédentaire"
  const candidatureFields = getCandidatureFields(isSedentaire)

  // Gestion du changement de statut avec logique métier
  const handleStatusChange = (name: string, value: any) => {
    const newStatus = value as CandidatureStatut
    const updatedValues = { ...formik.values, [name]: newStatus }

    // Si le statut est changé à Retenu ou Valide, s'assurer que dateHeureRdvAgence est défini
    if (
      (newStatus === CandidatureStatut.Retenu || newStatus === CandidatureStatut.Valide) &&
      !formik.values.dateHeureRdvAgence
    ) {
      // Définir une date par défaut pour le RDV (demain à 10h00)
      const tomorrow = new Date()
      tomorrow.setDate(tomorrow.getDate() + 1)
      tomorrow.setHours(10, 0, 0, 0)

      const dateString = tomorrow.toISOString().slice(0, 16) // Format: YYYY-MM-DDTHH:MM
      updatedValues.dateHeureRdvAgence = dateString
    }

    // Si le statut est changé à Valide ou Retenu, s'assurer que contrat est défini
    if ((newStatus === CandidatureStatut.Valide || newStatus === CandidatureStatut.Retenu) && !formik.values.contrat) {
      updatedValues.contrat = "non signé"
    }

    // Si le statut est changé à Valide, initialiser email_pro s'il n'est pas déjà défini
    if (newStatus === CandidatureStatut.Valide && !formik.values.email_pro) {
      // Créer un email professionnel basé sur l'email personnel ou le nom
      if (formik.values.email) {
        const emailParts = formik.values.email.split("@")
        if (emailParts.length > 1) {
          updatedValues.email_pro = emailParts[0] + "@votreentreprise.com"
        }
      } else if (formik.values.nom && formik.values.prenom) {
        const normalizedPrenom = formik.values.prenom
          .toLowerCase()
          .normalize("NFD")
          .replace(/[\u0300-\u036f]/g, "")
        const normalizedNom = formik.values.nom
          .toLowerCase()
          .normalize("NFD")
          .replace(/[\u0300-\u036f]/g, "")
        updatedValues.email_pro = `${normalizedPrenom}.${normalizedNom}@votreentreprise.com`
      }
    }

    // Si le statut est changé à Refuse ou Blackliste, s'assurer que raisonRefus est défini
    if (
      (newStatus === CandidatureStatut.Refuse || newStatus === CandidatureStatut.Blackliste) &&
      !formik.values.raisonRefus
    ) {
      updatedValues.raisonRefus = ""
    }

    // Mettre à jour les valeurs du formulaire
    Object.keys(updatedValues).forEach((key) => {
      formik.setFieldValue(key, updatedValues[key as keyof ICandidature])
    })
  }

  // Gestion du téléchargement de document
  const handleDocumentUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (file) {
      formik.setFieldValue("document", file.name)
    }
  }

  // Gestion du téléchargement de document de contrat
  const handleContratDocumentUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (file) {
      formik.setFieldValue("contrat_document", file.name)
    }
  }

  // Gestion du clic sur l'icône de document
  const handleDocumentIconClick = () => {
    if (!isViewMode) {
      documentFileInputRef.current?.click()
    }
  }

  // Gestion du clic sur l'icône de document de contrat
  const handleContratDocumentIconClick = () => {
    if (!isViewMode) {
      contratDocumentFileInputRef.current?.click()
    }
  }

  // Gestion de la modal d'email
  const handleOpenEmailModal = () => setIsEmailModalOpen(true)
  const handleCloseEmailModal = () => {
    setIsEmailModalOpen(false)
    setEmailSubject("")
    setEmailMessage("")
  }

  // Gestion de l'envoi d'email
  const handleSendEmail = () => {
    if (emailSubject && emailMessage) {
      // Logique pour envoyer un email au candidat
      console.log(`Sending email to ${formik.values.email}`, { subject: emailSubject, message: emailMessage })
      // Ici, vous appelleriez généralement une API pour envoyer l'email
      handleCloseEmailModal()
    }
  }

  // Gestion de la fermeture des notifications
  const handleCloseNotification = () => {
    setNotification({ ...notification, open: false })
  }

  const handleEditClick = () => {
    if (onEdit) {
      onEdit(candidature)
    }
  }

  return (
    <FormContainer
      titre={isNewMode ? "Nouvelle candidature" : `Candidature - ${formik.values.nom} ${formik.values.prenom}`}
    >
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" gap="8px" pb={1}>
        <ConditionalComponent isValid={isViewMode}>
          <EditExportButtons
            onEdit={handleEditClick}
            onExport={() => void 0}
            tooltipTitle={TableType.Candidature}
            dataRow={candidature}
            tableHead={tableHead}
          />
        </ConditionalComponent>
        <ConditionalComponent isValid={formik.values.statut === CandidatureStatut.ATraiter && isViewMode}>
          <Box
            sx={{
              borderLeft: 1,
              height: 30,
              borderColor: (theme) => theme.palette.text.disabled,
              mx: 1,
              mb: 1,
            }}
          />
        </ConditionalComponent>
        <ConditionalComponent isValid={formik.values.statut === CandidatureStatut.ATraiter}>
          <CustomTooltip arrow title="Envoyer un email">
            <AddButton color="primary" variant="contained" onClick={handleOpenEmailModal} sx={{ mr: 1, mb: 1 }}>
              Envoyer un email
            </AddButton>
          </CustomTooltip>
        </ConditionalComponent>
      </Stack>


      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={3}>
          {candidatureFields
            .filter((field) => {
              // Pour le statut ATraiter avec isSedentaire true, afficher uniquement des champs spécifiques
              if (formik.values.statut === CandidatureStatut.ATraiter && isSedentaire) {
                const aTraiterSedentaireFields = [
                  "nom",
                  "prenom",
                  "sourceCandidat",
                  "codePostal",
                  "ville",
                  "age",
                  "email",
                  "telephone",
                  "suiviAppel",
                  "dateAppelRappel",
                  "dateCandidature",
                  "document",
                  "departement",
                  "statut",
                  "address",
                ]
                return aTraiterSedentaireFields.includes(field.name)
              }

              // Pour le statut Retenu avec type chauffeur, afficher des champs spécifiques
              if (formik.values.statut === CandidatureStatut.Retenu && !isSedentaire) {
                const retenuChauffeurFields = [
                  "nom",
                  "prenom",
                  "sourceCandidat",
                  "email",
                  "telephone",
                  "dureePermis",
                  "departement",
                  "proximiteCircuit",
                  "contrat",
                  "document",
                  "codePostal",
                  "ville",
                  "age",
                  "statut",
                  "dateHeureRdvAgence",
                  "address",
                ]
                return retenuChauffeurFields.includes(field.name)
              }

              // Pour le statut Retenu avec type sedentaire, afficher des champs spécifiques
              if (formik.values.statut === CandidatureStatut.Retenu && isSedentaire) {
                const retenuSedentaireFields = [
                  "nom",
                  "prenom",
                  "codePostal",
                  "ville",
                  "age",
                  "telephone",
                  "email",
                  "dateHeureRdvAgence",
                  "departement",
                  "statut",
                  "contrat",
                  "address",
                ]
                return retenuSedentaireFields.includes(field.name)
              }

              // Pour le statut Valide, afficher uniquement les champs spécifiques demandés
              if (formik.values.statut === CandidatureStatut.Valide) {
                // Pour Valide avec type chauffeur
                if (!isSedentaire) {
                  const valideChauffeurFields = [
                    "nom",
                    "prenom",
                    "sourceCandidat",
                    "email",
                    "email_pro",
                    "telephone",
                    "dureePermis",
                    "departement",
                    "proximiteCircuit",
                    "contrat",
                    "document",
                    "contrat_document",
                    "codePostal",
                    "ville",
                    "age",
                    "statut",
                    "address",
                  ]
                  return valideChauffeurFields.includes(field.name)
                }
                // Pour Valide avec type sedentaire
                else {
                  const valideSedentaireFields = [
                    "nom",
                    "prenom",
                    "sourceCandidat",
                    "codePostal",
                    "ville",
                    "age",
                    "email",
                    "email_pro",
                    "telephone",
                    "contrat",
                    "document",
                    "contrat_document",
                    "departement",
                    "statut",
                    "address",
                  ]
                  return valideSedentaireFields.includes(field.name)
                }
              }

              // Pour le statut Absent, afficher uniquement les champs spécifiques demandés
              if (formik.values.statut === CandidatureStatut.Absent) {
                const absentFieldsStatus = [
                  "nom",
                  "prenom",
                  "sourceCandidat",
                  "codePostal",
                  "ville",
                  "age",
                  "email",
                  "telephone",
                  "dureePermis",
                  "dateHeureRdvAgence",
                  "departement",
                  "statut",
                ]
                return absentFieldsStatus.includes(field.name)
              }

              // Pour les stajtuts Refuse et Blackliste, afficher uniquement les champs spécifiques demandés
              if (
                formik.values.statut === CandidatureStatut.Refuse ||
                formik.values.statut === CandidatureStatut.Blackliste
              ) {
                const refuseBlacklisteFields = [
                  "nom",
                  "prenom",
                  "sourceCandidat",
                  "telephone",
                  "departement",
                  "email",
                  "raisonRefus",
                  "statut",
                  "address",
                ]
                return refuseBlacklisteFields.includes(field.name)
              }

              // Pour les autres statuts, utiliser la condition showWhen si fournie
              if (field.showWhen) {
                return field.showWhen(formik.values.statut)
              }

              return true
            })
            .map((field) => {
              // Changer le libellé pour dateHeureRdvAgence lorsque le statut est Absent
              if (field.name === "dateHeureRdvAgence" && formik.values.statut === CandidatureStatut.Absent) {
                const modifiedField = { ...field, label: "Replanifier un RDV" }
                return (
                  <Grid item xs={12} md={6} key={field.name}>
                    <FormField
                      field={modifiedField}
                      value={formik.values[field.name] || ""}
                      onChange={(name, value) => formik.setFieldValue(name, value)}
                      error={
                        formik.touched[field.name] && formik.errors[field.name]
                          ? (formik.errors[field.name] as string)
                          : undefined
                      }
                      onBlur={formik.handleBlur}
                      isReadOnly={isViewMode}
                    />
                  </Grid>
                )
              }

              // Changer le libellé pour raisonRefus lorsque le statut est Refuse ou Blackliste
              if (
                field.name === "raisonRefus" &&
                (formik.values.statut === CandidatureStatut.Refuse ||
                  formik.values.statut === CandidatureStatut.Blackliste)
              ) {
                const modifiedField = { ...field, label: "Raison", required: true }
                return (
                  <Grid item xs={12} md={6} key={field.name}>
                    <FormField
                      field={modifiedField}
                      value={formik.values[field.name] || ""}
                      onChange={(name, value) => formik.setFieldValue(name, value)}
                      error={
                        formik.touched[field.name] && formik.errors[field.name]
                          ? (formik.errors[field.name] as string)
                          : undefined
                      }
                      onBlur={formik.handleBlur}
                      isReadOnly={isViewMode}
                    />
                  </Grid>
                )
              }

              // Rendu personnalisé pour le champ document
              if (field.name === "document") {
                return (
                  <Grid item xs={12} md={6} key={field.name}>
                    <StyledTextField
                      isSmallScreen={isBetweenSmAndLg}
                      fullWidth
                      name={field.name}
                      InputLabelProps={{
                        shrink: true,
                      }}
                      value={formik.values[field.name] || ""}
                      onChange={formik.handleChange}
                      onBlur={formik.handleBlur}
                      label={field.label}
                      placeholder="Cliquez pour importer un document"
                      InputProps={{
                        readOnly: isViewMode,
                        endAdornment: (
                          <InputAdornment position="end">
                            <CustomTooltip arrow title="Importer un document">
                              <IconButton
                                onClick={handleDocumentIconClick}
                                disabled={isViewMode}
                                size="small"
                                color="primary"
                              >
                                <FontAwesomeIcon icon={faPaperclip} />
                              </IconButton>
                            </CustomTooltip>
                          </InputAdornment>
                        ),
                      }}
                    />
                  </Grid>
                )
              }

              // Rendu personnalisé pour le champ contrat_document
              if (field.name === "contrat_document") {
                return (
                  <Grid item xs={12} md={6} key={field.name}>
                    <StyledTextField
                      isSmallScreen={isBetweenSmAndLg}
                      fullWidth
                      name={field.name}
                      InputLabelProps={{
                        shrink: true,
                      }}
                      value={formik.values[field.name] || ""}
                      onChange={formik.handleChange}
                      onBlur={formik.handleBlur}
                      label={field.label}
                      placeholder="Cliquez pour importer le contrat"
                      InputProps={{
                        readOnly: isViewMode,
                        endAdornment: (
                          <InputAdornment position="end">
                            <CustomTooltip arrow title="Importer le contrat">
                              <IconButton
                                onClick={handleContratDocumentIconClick}
                                disabled={isViewMode}
                                size="small"
                                color="primary"
                              >
                                <FontAwesomeIcon icon={faPaperclip} />
                              </IconButton>
                            </CustomTooltip>
                          </InputAdornment>
                        ),
                      }}
                    />
                  </Grid>
                )
              }

              return (
                <Grid item xs={12} md={6} key={field.name}>
                  <FormField
                    field={field}
                    value={formik.values[field.name] || ""}
                    onChange={(name, value) =>
                      field.name === "statut" ? handleStatusChange(name, value) : formik.setFieldValue(name, value)
                    }
                    error={
                      formik.touched[field.name] && formik.errors[field.name] ?
                      field.name==="address" ? (formik.errors.address as { address?: string })?.address
                      :
                      String(formik.errors[field.name]) 
                        : undefined
                    }
                    onBlur={formik.handleBlur}
                    isReadOnly={isViewMode}
                  />
                </Grid>
              )
            })}
          <Grid item xs={12}>
            <ConditionalComponent isValid={!isViewMode}>
              <ActionButtons onSave={formik.submitForm} onCancel={() => onClose(false)} mode={mode} />
            </ConditionalComponent>
          </Grid>
        </Grid>
      </form>

      {/* Inputs de fichiers cachés */}
      <input
        type="file"
        ref={documentFileInputRef}
        style={{ display: "none" }}
        onChange={handleDocumentUpload}
        accept=".pdf,.doc,.docx,.jpg,.jpeg,.png"
      />
      <input
        type="file"
        ref={contratDocumentFileInputRef}
        style={{ display: "none" }}
        onChange={handleContratDocumentUpload}
        accept=".pdf,.doc,.docx,.jpg,.jpeg,.png"
      />

      <SendEmailModal
        open={isEmailModalOpen}
        onClose={handleCloseEmailModal}
        candidature={formik.values}
        onSend={handleSendEmail}
      />

      {/* Snackbar de notification */}
      <Snackbar
        open={notification.open}
        autoHideDuration={6000}
        onClose={handleCloseNotification}
        anchorOrigin={{ vertical: "top", horizontal: "center" }}
      >
        <Alert onClose={handleCloseNotification} severity={notification.severity} sx={{ width: "100%" }}>
          {notification.message}
        </Alert>
      </Snackbar>
    </FormContainer>
  )
}
