import {
    faClock,
    faListCheck,
    faCalendarDays
} from '@fortawesome/free-solid-svg-icons';
import PieCharts from '@/shared/components/charts/pie-charts';

import { Box, Grid, Card, Typography, alpha } from '@mui/material';
import { IDemandeAbsence } from '@/shared/types/leave';
import LeaveBalanceCard from './leave-balance-card';


interface AbsenceOverviewProps {
    data: IDemandeAbsence[];
}

export default function AbsenceOverview({ data }: AbsenceOverviewProps) {
    // Calculate actual counts from data
    const totalAbsences = data.length;
    const statusCounts = {
        enAttente: data.filter(item => item.status === 'En attente').length,
        validees: data.filter(item => item.status === 'Approuvée').length,
        refusees: data.filter(item => item.status === 'Refusée').length
    };
    
    
    const typeCounts = {
        conge: data.filter(item => item.DemandeType === 'Congé').length,
        absence: data.filter(item => item.DemandeType === 'Absence').length
    };

    const stats = [
        {
            title: 'Total Demandes',
            total: 30,
            icon: faListCheck,
            color: '#5176FE',
        },
        {
            title: 'Total Absences',
            total: 10,
            icon: faClock,
            color: '#F9B90B',
        },
        {
            title: 'Total Congés',
            total: 20,
            icon: faCalendarDays,
            color: '#36B37E',
        },
    ];

    // Single chart for status distribution
    const charts = [
        {
            title: 'Répartition par statut',
            data: [
                {
                    id: 'En attente',
                    value: statusCounts.enAttente || 10,
                    label: 'En attente',
                    color: '#F9B90B'
                },
                {
                    id: 'Validé',
                    value: statusCounts.validees || 10,
                    label: 'Validé',
                    color: '#36B37E'
                },
                {
                    id: 'Refusé',
                    value: statusCounts.refusees || 10,
                    label: 'Refusé',
                    color: '#FF3434'
                },
            ],
        },
    ];
    const balances = [
        {
            type: 'Annuel',
            used: 5,
            total: 10,
            color: '#FF6B6B',
        },
        {
            type: 'Maladie',
            used: 3,
            total: 10,
            color: '#4ECDC4',
        },
        {
            type: 'Parental',
            used: 2,
            total: 10,
            color: '#45B7D1',
        },
        {
            type: 'RTT',
            used: 1,
            total: 10,
            color: '#96CEB4',
        },
    ];
    return (
        <Box sx={{ width: '100%' }}>
            {/* Single row layout with all elements */}
            <Grid container spacing={2}>
                {/* Status Distribution Chart */}
                <Grid item xs={12} md={4}>
                    <Card
                        sx={{
                            p: 2,
                            height: '100%',
                            borderRadius: 2,
                            border: (theme) => `1px solid ${theme.palette.divider}`,
                            boxShadow: 'none',
                            bgcolor: 'background.paper',
                            transition: 'all 0.3s ease',
                            '&:hover': {
                                boxShadow: '0 4px 20px 0 rgba(0,0,0,0.05)',
                                bgcolor: (theme) => alpha(theme.palette.primary.main, 0.02),
                                borderColor: (theme) => alpha(theme.palette.primary.main, 0.2),
                            }
                        }}
                    >
                        <Typography
                            variant="h6"
                            sx={{
                                mb: 2,
                                color: (theme) => theme.palette.primary.main,
                                fontWeight: 600
                            }}
                        >
                            Répartition par statut
                        </Typography>
                        <Box sx={{ display: 'flex', justifyContent: 'center', alignItems: 'center', height: 'calc(100% - 40px)' }}>
                            <PieCharts charts={charts} fullWidth />
                        </Box>
                    </Card>
                </Grid>

                {/* Leave Balance Card */}
                <Grid item xs={12} md={4}>
                    <LeaveBalanceCard balances={balances} />
                </Grid>

                {/* Stats Cards in a vertical stack with consistent styling */}
                <Grid item xs={12} md={4}>
                    <Card
                        sx={{
                            p: 2,
                            height: '100%',
                            borderRadius: 2,
                            border: (theme) => `1px solid ${theme.palette.divider}`,
                            boxShadow: 'none',
                            bgcolor: 'background.paper',
                            transition: 'all 0.3s ease',
                            '&:hover': {
                                boxShadow: '0 4px 20px 0 rgba(0,0,0,0.05)',
                                bgcolor: (theme) => alpha(theme.palette.primary.main, 0.02),
                                borderColor: (theme) => alpha(theme.palette.primary.main, 0.2),
                            }
                        }}
                    >
                        <Typography
                            variant="h6"
                            sx={{
                                mb: 2,
                                color: (theme) => theme.palette.primary.main,
                                fontWeight: 600
                            }}
                        >
                            Statistiques
                        </Typography>
                        <Grid container spacing={1} sx={{ height: 'calc(100% - 40px)' }}>
                            {stats.map((stat, index) => (
                                <Grid item xs={12} key={index}>
                                    <Box
                                        sx={{
                                            display: 'flex',
                                            alignItems: 'center',
                                            p: 1,
                                            borderRadius: 1,
                                            border: '1px solid',
                                            borderColor: alpha(stat.color, 0.2),
                                            bgcolor: alpha(stat.color, 0.05),
                                            transition: 'all 0.3s ease',
                                            '&:hover': {
                                                boxShadow: '0 2px 10px 0 rgba(0,0,0,0.05)',
                                                bgcolor: alpha(stat.color, 0.1),
                                                borderColor: alpha(stat.color, 0.3),
                                                transform: 'translateY(-2px)'
                                            },
                                            cursor: 'pointer'
                                        }}
                                    >
                                        <Box
                                            sx={{
                                                width: 40,
                                                height: 40,
                                                display: 'flex',
                                                alignItems: 'center',
                                                justifyContent: 'center',
                                                borderRadius: '50%',
                                                backgroundColor: alpha(stat.color, 0.1),
                                                color: stat.color,
                                                mr: 2,
                                                transition: 'all 0.3s ease',
                                                '&:hover': {
                                                    backgroundColor: alpha(stat.color, 0.2),
                                                }
                                            }}
                                        >
                                            <Box component="span" sx={{ fontSize: 20, color: stat.color }}>
                                                {stat.total}
                                            </Box>
                                        </Box>
                                        <Typography variant="body1" sx={{ fontWeight: 'medium' }}>
                                            {stat.title}
                                        </Typography>
                                    </Box>
                                </Grid>
                            ))}
                        </Grid>
                    </Card>
                </Grid>
            </Grid>
        </Box>
    );

}