import React from 'react';
import { Box, Typography, Card, LinearProgress, Grid, alpha } from '@mui/material';
import { styled } from '@mui/material/styles';

// Define types
interface LeaveBalance {
  type: string;
  used: number;
  total: number;
  color: string;
}

interface LeaveBalanceCardProps {
  balances: LeaveBalance[];
}

// Styled components
const BalanceCard = styled(Card)(({ theme }) => ({
  padding: theme.spacing(2),
  borderRadius: 12,
  border: `1px solid ${theme.palette.divider}`,
  boxShadow: 'none',
  backgroundColor: theme.palette.background.paper,
  height: '100%',
  display: 'flex',
  flexDirection: 'column',
  justifyContent: 'space-between',
  transition: 'all 0.3s ease',
  '&:hover': {
    boxShadow: '0 4px 20px 0 rgba(0,0,0,0.05)',
    backgroundColor: alpha(theme.palette.primary.main, 0.02),
    borderColor: alpha(theme.palette.primary.main, 0.2),
  }
}));

const ProgressLabel = styled(Box)({
  display: 'flex',
  justifyContent: 'space-between',
  marginBottom: 0,
});

export default function LeaveBalanceCard({ balances }: LeaveBalanceCardProps) {
  return (
    <Grid container sx={{ height: '100%' }}>
      <Grid item xs={12}>
        <BalanceCard>
          <Typography variant="h6" fontWeight={600} sx={{ mb: 1.5, color: (theme) => theme.palette.primary.main }}>
            Solde de congés restants
          </Typography>
          {balances.map((balance) => {
            const remaining = balance.total - balance.used;
            const percentUsed = (balance.used / balance.total) * 100;

            let statusColor = balance.color;
            if (balance.type === 'Sans solde') {
              statusColor = '#FF6B6B'; // Red for unpaid leave
            } else if (remaining <= balance.total * 0.2) {
              statusColor = '#FF3434'; // Red for low balance
            } else if (remaining <= balance.total * 0.5) {
              statusColor = '#FFAB00'; // Yellow for medium balance
            }

            return (
              <Box 
                key={balance.type} 
                sx={{ 
                  mb: 1.5,
                  p: 0.5,
                  borderRadius: 1,
                  cursor: 'pointer',
                  transition: 'all 0.2s',
                  '&:hover': {
                    bgcolor: (theme) => alpha(theme.palette.primary.main, 0.05),
                    transform: 'translateX(4px)',
                    boxShadow: '0 2px 8px rgba(0,0,0,0.05)',
                  },
                }}
              >
                <ProgressLabel>
                  <Typography variant="body2" fontWeight={500} color="text.primary" fontSize="0.8rem">
                    {balance.type}
                  </Typography>
                  <Typography variant="body2" fontWeight={700} color="text.primary" fontSize="0.8rem">
                    {remaining} jours
                  </Typography>
                </ProgressLabel>

                <Box sx={{ position: 'relative' }}>
                  <LinearProgress
                    variant="determinate"
                    value={balance.type === 'Sans solde' ? 100 : percentUsed}
                    sx={{
                      height: 5,
                      borderRadius: 2,
                      backgroundColor: '#EAECEF',
                      transition: 'all 0.3s',
                      '& .MuiLinearProgress-bar': {
                        borderRadius: 2,
                        backgroundColor: statusColor,
                      }
                    }}
                  />

                  {balance.type === 'Sans solde' && (
                    <Typography
                      variant="caption"
                      sx={{
                        position: 'absolute',
                        top: '50%',
                        left: '50%',
                        transform: 'translate(-50%, -50%)',
                        color: '#FFF',
                        fontWeight: 'bold',
                        fontSize: '0.65rem'
                      }}
                    >
                      Illimité
                    </Typography>
                  )}
                </Box>
              </Box>
            );
          })}
        </BalanceCard>
      </Grid>
    </Grid>
  );
} 