'use client';

import React, { useEffect } from 'react';
import {
  Typography,
  Grid,
  Box,
} from '@mui/material';
import { LocalizationProvider } from '@mui/x-date-pickers/LocalizationProvider';
import { AdapterDateFns } from '@mui/x-date-pickers/AdapterDateFns';
import { fr } from 'date-fns/locale';
import { useFormik } from 'formik';
import FormField from '@/shared/components/form/form-field';
import { FormFieldType, ModeType } from '@/shared/types/common';
import FormContainer from '@/layouts/form/form-container';
import EditExportButtons from '@/shared/components/form/edit-export-buttons';
import ActionButtons from '@/shared/components/form/buttons-action';
import { AddButton } from '@/shared/components/table/styles';
import FontAwesome from '@/shared/components/fontawesome';
import { faFileArrowUp } from '@fortawesome/free-solid-svg-icons';
import { IDemandeAbsence } from '@/shared/types/leave';
import { absenceFormFields } from '../utils/form-fields-absence';
import { absenceSchema } from '../utils/absence-schema';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

interface AbsenceFormProps {
  absence: IDemandeAbsence;
  mode: ModeType;
  onSave: (updatedAbsence: IDemandeAbsence) => void;
  onEdit: (updatedAbsence: IDemandeAbsence) => void;
  onClose: (forceClose: boolean) => void;
  updateTabContent: (tabId: string, newContent: IDemandeAbsence) => void
  tabId: string
}

export default function AbsenceForm({
  absence,
  mode,
  onSave,
  onClose,
  onEdit,
  updateTabContent,
  tabId,
}: AbsenceFormProps) {
  const isReadOnly = mode === 'view';

  const formik = useFormik<IDemandeAbsence>({
    initialValues: {
      ...absence,
      DemandeType: absence?.DemandeType || 'Congé',
      AbsenceType: absence?.AbsenceType || 'Formation professionnelle' as any,
      leaveType: absence?.leaveType || 'Congé annuel' as any
    },
    validationSchema: absenceSchema,
    onSubmit: (values) => {
      onSave(values);
      onClose(true);
    },
  });

  useEffect(() => {
    if (updateTabContent && tabId && (mode !== "view")) {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const getFormTitle = () => {
    switch (mode) {
      case 'view':
        return 'Fiche de Demande';
      case 'edit':
        return 'Modification Demande';
      case 'add':
        return 'Nouvelle Demande';
      default:
        return 'Demande';
    }
  };

  const renderField = (field: FormFieldType<IDemandeAbsence>) => (
    <FormField
      field={field}
      value={formik.values[field.name as keyof IDemandeAbsence]}
      onChange={(name, value) => {
        formik.setFieldValue(name, value);
        if (name === 'DemandeType') {
          if (value === 'Absence') {
            formik.setFieldValue('AbsenceType', 'Formation professionnelle');
            formik.setFieldValue('leaveType', '');
          } else {
            formik.setFieldValue('leaveType', 'Congé annuel');
            formik.setFieldValue('AbsenceType', '');
          }
        }
      }}
      error={
        formik.touched[field.name as keyof IDemandeAbsence]
          ? (formik.errors[field.name as keyof IDemandeAbsence] as string | undefined)
          : undefined
      }
      onBlur={formik.handleBlur}
      isReadOnly={isReadOnly}
    />
  );

  return (
    <LocalizationProvider dateAdapter={AdapterDateFns} adapterLocale={fr}>
      <FormContainer titre={getFormTitle()}>
        <ConditionalComponent isValid={isReadOnly}>
          <EditExportButtons
            onEdit={() => onEdit(formik.values)}
            onExport={() => void 0}
            tooltipTitle="l'absence"
          />
        </ConditionalComponent>

        <form onSubmit={formik.handleSubmit}>
          <Grid container spacing={4}>
            <Grid item xs={12} md={4}>
              {renderField(absenceFormFields.find(f => f.name === 'DemandeType')!)}
            </Grid>
            <ConditionalComponent isValid={formik.values.DemandeType === 'Absence'} 
            defaultComponent={
              <Grid item xs={12} md={8}>
                {renderField(absenceFormFields.find(f => f.name === 'leaveType')!)}
              </Grid>
            }
            >
              <Grid item xs={12} md={8}>
                {renderField(absenceFormFields.find(f => f.name === 'AbsenceType')!)}
              </Grid>
            </ConditionalComponent>

            <Grid item xs={12} md={4}>
              {renderField(absenceFormFields.find(f => f.name === 'startDate')!)}
            </Grid>
            <Grid item xs={12} md={4}>
              {renderField(absenceFormFields.find(f => f.name === 'endDate')!)}
            </Grid>
            <Grid item xs={12} md={4}>
              {renderField(absenceFormFields.find(f => f.name === 'daysCount')!)}
            </Grid>
            <Grid item xs={12} md={4}>
              {renderField(absenceFormFields.find(f => f.name === 'status')!)}
            </Grid>
            <Grid item xs={12} md={8}>
              {renderField(absenceFormFields.find(f => f.name === 'reason')!)}
            </Grid>

            <Grid item xs={12}>
              <Box display="flex" alignItems="center" gap={4}>
                <Typography sx={{ color: 'primary.main' }}>
                  Pièce justificative
                </Typography>
                <AddButton
                  variant="contained"
                  sx={{ minHeight: '43px' }}
                  endIcon={<FontAwesome icon={faFileArrowUp} width={18} />}
                  disabled={isReadOnly}
                >
                  Importez un document
                </AddButton>
              </Box>
            </Grid>
            <Grid item xs={12}>
              <ConditionalComponent isValid={!isReadOnly}>
                <ActionButtons
                onSave={() => {
                  formik.validateForm().then((errors) => {
                    formik.submitForm()
                  })
                }}
                  onCancel={() => onClose(false)}
                  mode={mode}
                />
              </ConditionalComponent>
            </Grid>
          </Grid>
        </form>
      </FormContainer>
    </LocalizationProvider>
  );
}
