import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import SvgColor from '@/shared/components/svg-color';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import { ContratStyledRow, getStatusColor, StatusChip } from '../../contract/styles';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { _CIRCUIT_STATUS } from '@/shared/_mock/_circuit';
import StatusMenu from '@/shared/components/table/StatusMenu';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { _PassengerStatus, IPassengerList } from '@/shared/types/passenger';
import { usePassengerStore } from '@/shared/api/stores/passengerStore';
import { PassengerStatusChip } from '@/shared/theme/css';
import { getLabel } from '@/shared/_mock';
import { _classesOptions, _RELATION_TYPE_OPTIONS } from '../utils/form-fields-usager';

type Props = {
  handleEdit: (usager: IPassengerList) => void;
  handleDetails: (usager: IPassengerList) => void;
  handleCopy: (usager: IPassengerList) => void;
  row: IPassengerList;
  dense: boolean;
  columns: TableColumn[];
  onStatusChange: (row: IPassengerList, newStatus: string, field: keyof IPassengerList) => void;
  
};
const TOOLTIP_TITLES = {
  edit: 'Modifier usager',
  duplicate:"Copier usager",
  active:"usager Actif",
  inactive:"usager Inactif",
  };
export default function UsagerTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  columns,
  onStatusChange
}: Props) {
  const { toggleArchive, passengers } = usePassengerStore();

  const handleToggleActivation = async () => {
    await toggleArchive(row.id);
    row.archived = !row.archived;
  }
  
  const renderCellContent = (columnId: string, value: any) => {
    if (value === null) {
      return '-';
    }

    switch (columnId) {
      case 'landlinePhones':
        return row.representatives?.length ? row.representatives[0]?.contacts[0]?.landlinePhones : '-';
      case 'mobilePhones':
        return row.representatives?.length ? row.representatives[0]?.contacts[0]?.mobilePhones : '-';
      case 'code':
        return row.representatives?.length ? row.representatives[0]?.circuitCode : '-';
      case 'address':
        return row.representatives?.length ? row.representatives[0]?.addresses[0]?.additionalAddress : '-';
      case 'code':
        return row.representatives?.length ? row.representatives[0]?.circuitCode : '-';
      case 'type':
        return  row.representatives?.length ? getLabel(row.representatives[0]?.type,_RELATION_TYPE_OPTIONS) : '-';
      case 'schoolClass':
        return  getLabel(value,_classesOptions);
      case 'responsibleName':
        return row.representatives?.length ? row.representatives[0]?.lastName : '-';
      case 'responsiblePrenom':
        return row.representatives?.length ? row.representatives[0]?.firstName : '-';
      case 'birthDate':
      case 'startTransport':
        return formatDate(value, dateFormat.isoDate);
      case 'state':
        return (
          <StatusMenu
            value={value}
            options={_PassengerStatus}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'state')} 
            chipComponent={PassengerStatusChip}
          />              
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>
      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'etat' ? 0.2 : dense ? 0 : 1, ...(index === 0 && {
              position: 'relative',
            }), }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: 'absolute', top: 0, left: -4, height: '100%', color: getStatusColor(row.state),
                }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IPassengerList])}
          </TableCell>
        ))}
      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEdit(row)}
          onDuplicate={() => handleCopy(row)}
          onToggle={() => handleToggleActivation()}
          isActive={row.archived}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}
