import { Autocomplete, Chip, FormControl, InputLabel, ListItemIcon, MenuItem, RadioGroup, Select, Tab, TableContainer, Tabs, TextField, TextFieldProps } from '@mui/material';
import { alpha } from '@mui/material/styles';
import { Box, styled } from '@mui/system';
import { DatePicker, TimePicker } from '@mui/x-date-pickers';
import { lightColor } from '../components/table/styles';
import { pxToRem } from './typography';
import { ExportTab } from '../types/export';
import { State } from '../types/passenger';

type BgBlurProps = {
  blur?: number;
  opacity?: number;
  color?: string;
  imgUrl?: string;
};

export function bgBlur(props?: BgBlurProps) {
  const color = props?.color || '#746CD4';
  const blur = props?.blur || 6;
  const opacity = props?.opacity || 0.8;
  const imgUrl = props?.imgUrl;

  if (imgUrl) {
    return {
      position: 'relative',
      backgroundImage: `url(${imgUrl})`,
      '&:before': {
        position: 'absolute',
        top: 0,
        left: 0,
        zIndex: 9,
        content: '""',
        width: '100%',
        height: '100%',
        backdropFilter: `blur(${blur}px)`,
        WebkitBackdropFilter: `blur(${blur}px)`,
        backgroundColor: alpha(color, opacity),
      },
    } as const;
  }

  return {
    backdropFilter: `blur(${blur}px)`,
    WebkitBackdropFilter: `blur(${blur}px)`,
    backgroundColor: alpha(color, opacity),
  };
}
export const noWrapStyles = {
  width: '100%',
  maxWidth: '100%',
  display: 'block',
  overflow: 'hidden',
  textOverflow: 'ellipsis',
  textTransform: 'capitalize',
} as const;

export const StyledTab = styled(Tab)(({ theme }) => ({
  minHeight: '35px',
  textTransform: 'none',
  padding: '0 10px',
  borderTopRightRadius: '12px',
  borderTopLeftRadius: '12px',
  borderBottom: `2px solid ${theme.palette.primary.main}`,
  backgroundColor: 'white',
  maxWidth: '500px',
  '&.Mui-selected': {
    backgroundColor: theme.palette.primary.main,
    color: theme.palette.primary.contrastText,
  },
  [theme.breakpoints.down('lg')]: {
    fontSize: '13px',
    minWidth: 100,
  },
}));

export const tabsStyles = {
    flexGrow: 1,
    "& .MuiTabs-indicator": { display: "none" },
    minHeight: "35px",
    "& .MuiTabs-flexContainer": {
      gap: "4px",
    },
}

export const StyledListItemIcon = styled(ListItemIcon)(({ theme }) => ({
  color: theme.palette.primary.main,
}))

export const StyledFilterTabs = styled(Tabs)(({ theme }) => ({
  mb: 2,
  minHeight: "40px",
  "& .MuiTab-root": {
    textTransform: "none",
    minWidth: "unset",
    color: alpha("#617581", 0.8),
    minHeight: "40px",
    padding: "0 16px",
    lineHeight: 1.2,
  },
  "& .Mui-selected": {
    color: theme.palette.primary.main,
  },
}));


export const StyledMenuItem = styled(MenuItem)(({ theme }) => ({
  fontSize: '0.9rem',
  borderRadius:'6px',
  marginInline:'8px',
  marginTop:'4px',
  '&:hover': {
    backgroundColor: alpha(theme.palette.primary.main,0.1),
    color:theme.palette.primary.main,
  },
  '&.Mui-selected': {
    backgroundColor: theme.palette.primary.main,
    color: 'white',
    '&:hover': {
      backgroundColor: theme.palette.primary.main,
    },
  },
}));
export const StyledInputLabel = styled(InputLabel)(({ theme }) => ({
  '&.MuiInputLabel-shrink': {
    transform: 'translate(14px, 4px) scale(0.8)',
  },
  '&.MuiFormLabel-root': {
    color:theme.palette.primary.main,
    transform: 'translate(14px, 4px) scale(0.8)',
  },
}))


type StyledTextFieldProps = TextFieldProps & {
  isSmallScreen?: boolean
  isReadOnly?:boolean
}

export const StyledTextField = styled(TextField, {
  shouldForwardProp: (prop) => 
    !['isSmallScreen', 'isReadOnly'].includes(prop.toString()),
})<StyledTextFieldProps>(({ theme, multiline, isSmallScreen, isReadOnly }) => ({
  "& .MuiOutlinedInput-root": {
    backgroundColor: alpha(theme.palette.primary.main, 0.03),
    paddingBottom: multiline ? "7px" : "0px",
    ...(isReadOnly && {
      backgroundColor: '#f5f5f5',
      cursor: 'not-allowed',
      "& .MuiInputBase-input": {
        cursor: 'not-allowed',
        color: theme.palette.text.disabled,
      }
    }),
    "& .MuiOutlinedInput-input": {
      marginTop: multiline ? 0 : "16px",
      marginBottum: multiline ? 0 : "16px",
      padding: multiline ? '10px 14px 0 0' : '10px 14px 8px',
      ...(isSmallScreen && {
        fontSize: pxToRem(13),
      }),
    },
    "&:hover .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
    },
    "& .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
      borderRadius: "12px",
      "& legend": {
        width: 0,
      },
    },
    "&.Mui-focused .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
      boxShadow: `0 0 0 4px ${alpha(theme.palette.primary.main, 0.1)}`,
    },
  },
  "& .MuiInputLabel-root": {
    transform: "translate(14px, 4px) scale(0.8)",
    color: theme.palette.primary.main,
  },
  "& .MuiInputLabel-asterisk": {
    color: "#FF0000",
  },
}))



const BaseStyledSelect = styled(Select)(({ theme }) => ({
  "& .MuiSvgIcon-root": {
    color: theme.palette.primary.main,
  },
  "& .MuiOutlinedInput-notchedOutline": {
    borderColor: theme.palette.primary.main,
  },
  "&:hover .MuiOutlinedInput-notchedOutline": {
    borderColor: theme.palette.primary.main,
  },
  "&.Mui-focused .MuiOutlinedInput-notchedOutline": {
    borderColor: theme.palette.primary.main,
  },
}))

export const StyledSelect = styled(BaseStyledSelect)<{isReadOnly?: boolean}>(
  ({ theme, isReadOnly }) => ({
    backgroundColor: isReadOnly 
      ? '#f5f5f5' 
      : alpha(theme.palette.primary.main, 0.03),

    "& .MuiOutlinedInput-input": {
      marginTop: "16px",
      padding: "10px 14px 8px",
      ...(isReadOnly && {
      cursor: 'not-allowed',
      })
    },
    "& .MuiOutlinedInput-notchedOutline": {
      borderRadius: "12px",
    },
    ...(isReadOnly && {
      cursor: 'not-allowed',
      '& .MuiSelect-select': {
        color: 'text.disabled',
      },
      '& .MuiSvgIcon-root': {
        color: 'text.disabled',
      },
    }),
  })
);


export const RoadMapStyledSelect = styled(BaseStyledSelect)(({ theme }) => ({
  backgroundColor: "white",
  minWidth:'200px',
  borderRadius: "8px",
  "& .MuiOutlinedInput-input": {
    padding: "9px 14px 10px",
  },
}))

export const StyledFormControl = styled(FormControl)({
  '& .MuiInputLabel-asterisk': {
    color: '#FF0000',
  },
});
export const ScrollBox = styled(Box)(({ theme }) => ({
  overflowY: 'auto',
  '&::-webkit-scrollbar': {
    width: '8px',
    background: 'transparent',
  },
  '&::-webkit-scrollbar-thumb': {
    backgroundColor: 'transparent',
    borderRadius: '3px',
    transition: 'background-color 0.2s',
  },
  '&:hover::-webkit-scrollbar-thumb': {
    backgroundColor: theme.palette.grey[400],
  },
}));

export const iconButtonStyles = {
  padding:'6px',
  ...lightColor
};
export const ActionsIconButton = {
  color: (theme: any) => theme.palette.text.primary,
  '&:hover': {
    color: (theme: any) => theme.palette.primary.main,
  },
};
export const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  scrollbarWidth: 'none',
  msOverflowStyle: 'none',
  '&::-webkit-scrollbar': {
    display: 'none',
  },
  '&:hover': {
    scrollbarWidth: 'thin',
    scrollbarColor: `${theme.palette.grey[400]} ${theme.palette.grey[100]}`,
    '&::-webkit-scrollbar': {
      display: 'block',
    },
    '&::-webkit-scrollbar-track': {
      background: theme.palette.grey[100],
    },
    '&::-webkit-scrollbar-thumb': {
      backgroundColor: theme.palette.grey[300],
      borderRadius: '3px',
      '&:hover': {
        backgroundColor: theme.palette.grey[400],
      },
    },
  },
}));

export const centerFlexColumn = {
  display: 'flex',
  flexDirection: 'column',
  justifyContent: 'center',
};

const sharedStyles = (theme: any, isSmallScreen?: boolean) => ({
  "& .MuiOutlinedInput-root": {
    backgroundColor: alpha(theme.palette.primary.main, 0.02),
    borderRadius: "12px",
    "& .MuiOutlinedInput-input": {
      marginTop: "16px",
      padding: "10px 14px 8px",
      fontSize: isSmallScreen ? pxToRem(13) : "inherit",
    },
    "& .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
      "& legend": {
        width: 0,
      },
    },
    "&:hover .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
    },
  },
  "& .MuiInputLabel-root": {
    transform: "translate(14px, 4px) scale(0.8)",
    color: theme.palette.primary.main,
  },
  "& .MuiInputLabel-asterisk": {
    color: "#FF0000",
  },
})

export const StyledAutocomplete = styled(Autocomplete, {
  shouldForwardProp: (prop) => prop !== "isSmallScreen",
})<{ isSmallScreen?: boolean }>(({ theme, isSmallScreen }) => ({
  "& .MuiInputBase-root": {
    backgroundColor: alpha(theme.palette.primary.main, 0.03),
    borderRadius: "12px",
    height: "56px",
    "&:hover": {
      backgroundColor: alpha(theme.palette.primary.main, 0.08),
      color: theme.palette.primary.main,
      "& .MuiOutlinedInput-notchedOutline": {
        borderColor: theme.palette.primary.main,
      },
    },
    "& .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
      borderRadius: "12px",
      "& legend": {
        width: 0,
      },
    },
    "&.Mui-focused .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
      boxShadow: `0 0 0 4px ${alpha(theme.palette.primary.main, 0.1)}`,
    },
  },
  "& .MuiAutocomplete-inputRoot": {
    padding: "0 14px !important",
  },
  "& .MuiAutocomplete-input": {
    padding: "7.5px 0 !important",
    fontSize: isSmallScreen ? pxToRem(13) : "inherit",
  },
  "& .MuiAutocomplete-endAdornment": {
    display: "none",
  },
  "& .MuiAutocomplete-paper": {
    borderRadius: "8px",
    marginTop: "4px",
  },


}));
export const RoadMapStyledDatePicker = styled(DatePicker)(({ theme }) => ({
  ...sharedStyles(theme),
  width:'200px',
  "& .MuiOutlinedInput-root": {
    backgroundColor: "white",
    borderRadius: "8px",
    "& .MuiOutlinedInput-input": {
      marginTop: 0,
      padding: "9px 14px 10px",
      fontSize: "inherit",
    },
    "& .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
      "& legend": {
        width: 0,
      },
    },
  },
}))

export const StyledTimePicker = styled(TimePicker)(({ theme, disabled }) => ({
  ...sharedStyles(theme),
  ...(disabled && {
    '& .MuiInputBase-root': {
      backgroundColor: '#f5f5f5',
      cursor: 'not-allowed',
    },
    '& .MuiInputBase-input': {
      cursor: 'not-allowed',
      color: `${theme.palette.text.disabled}`,
    },
    '& .MuiOutlinedInput-input': {
      color: `${theme.palette.text.disabled}`,
    }
  })
}));

export const RoadMapStyledTimePicker = styled(TimePicker)(({ theme, disabled }) => ({
  ...sharedStyles(theme),
  "& .MuiOutlinedInput-root": {
    backgroundColor: alpha(theme.palette.primary.main, 0.02),
    borderRadius: "8px",
    "& .MuiOutlinedInput-input": {
      marginTop:"10px",
      padding: "9px 14px 6px",
      fontSize: "inherit",
    },
    "& .MuiOutlinedInput-notchedOutline": {
      borderColor: theme.palette.primary.main,
      "& legend": {
        width: 0,
      },
    },
  },
  ...(disabled && {
    '& .MuiInputBase-root': {
      backgroundColor: '#f5f5f5',
      cursor: 'not-allowed',
    },
    '& .MuiInputBase-input': {
      cursor: 'not-allowed',
      color: `${theme.palette.text.disabled}`,
    },
    '& .MuiOutlinedInput-input': {
      color: `${theme.palette.text.disabled}`,
    }
  })
}));
export const StyledRadioGroup = styled(RadioGroup)(({ theme }) => ({
  flexDirection: 'row',
  gap: theme.spacing(2),
  '& .MuiFormControlLabel-root': {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: '6px 12px',
    border: `1px solid ${theme.palette.primary.main}`,
    margin: 0,
  },
  '& .MuiRadio-root': {
    padding: 4,
  },
}));

export const StyledDatePicker = styled(DatePicker, {
  shouldForwardProp: (prop) => prop !== "isSmallScreen",
})<{ isSmallScreen?: boolean }>(({ theme, isSmallScreen, readOnly }) => ({
  ...sharedStyles(theme, isSmallScreen),
  ...(readOnly && {
    '& .MuiInputBase-root': {
      backgroundColor: '#f5f5f5',
      cursor: 'not-allowed',
    },
    '& .MuiInputBase-input': {
      cursor: 'not-allowed',
      color: `${theme.palette.text.disabled}`,
    },
    '& .MuiOutlinedInput-input': {
      color: `${theme.palette.text.disabled}`,
    }
  })
}))

export const WhiteIconButtonStyle = {
  color : alpha("#617581",0.8),
  bgcolor: (theme: any) => theme.palette.background.default,
  '&:hover': {
    bgcolor: (theme: any) => theme.palette.action.hover,
  },
};

export const tabsBorder = {
  pb: 2,
  px: 2,
  borderRadius: "0px 16px 16px 16px",
  border: (theme: any)  => `2px solid ${theme.palette.primary.main}`,
};

export const dialogTitleStyles = (theme: any) => ({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
  backgroundColor: theme.palette.primary.main,
  color: 'white',
  fontWeight: 600,
  padding: '16px 24px',
  borderBottom: `1px solid ${theme.palette.divider}`,
});

export const dialogTitleTextStyles = {
  fontSize: '1.25rem',
  fontWeight: 'bold',
  color: 'white',
};

export const baseChipStyles = {
  fontWeight: 500,
  borderRadius: "8px",
  padding: "9px 0",
  height: "100%",
  width: "100%",
}

export const StatusChipExport = styled(Chip)(({ status, theme }: { status: ExportTab; theme: any }) => ({
  '&.MuiChip-root': {
    ...baseChipStyles,
    ...(status === ExportTab.COMPLETED && {
      backgroundColor: alpha(theme.palette.success.main, 0.1),
      color: theme.palette.success.main,
    }),
    ...(status === ExportTab.IN_PROGRESS && {
      backgroundColor: alpha(theme.palette.warning.main, 0.1),
      color: theme.palette.warning.main,
    }),
    ...(status === ExportTab.FAILED && {
      backgroundColor: alpha(theme.palette.error.main, 0.1),
      color: theme.palette.error.main,
    }),
  },
}));

export const PassengerStatusChip = styled(Chip)(({ status, theme }: { status: string; theme: any }) => ({
  '&.MuiChip-root': {
    ...baseChipStyles,
    ...(status === State.CONTROLLED && {
      backgroundColor: alpha(theme.palette.success.main, 0.1),
      color: theme.palette.success.main,
    }),
    ...(status === State.UNCONTROLLED && {
      backgroundColor: alpha(theme.palette.error.main, 0.1),
      color: theme.palette.error.main,
    }),
  },
}));
