import { Box, Card, Stack, Typography, Grid, alpha } from '@mui/material';
import { PieChart } from '@mui/x-charts';
import { ScrollBox } from '@/shared/theme/css';

interface ChartData {
  id: string;
  value: number;
  label: string;
  color: string;
}

interface ChartSection {
  title: string;
  data: ChartData[];
}

interface PieChartsProps {
  charts: ChartSection[];
  fullWidth?: boolean;
}

export default function PieCharts({ charts, fullWidth = false }: PieChartsProps) {
  if (fullWidth) {
    return (
      <Box sx={{ width: '100%', display: 'flex', justifyContent: 'center' }}>
        {charts.map((chart, index) => (
          <Box 
            key={index}
            sx={{ 
              display: 'flex', 
              flexDirection: 'row',
              alignItems: 'center',
              justifyContent: 'space-between',
              width: '100%',
            }}
          >
            <Box 
              sx={{ 
                width: '40%', 
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
              }}
            >
              <Box 
                sx={{ 
                  position: 'relative',
                  width: 150,
                  height: 150,
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center',
                }}
              >
                <PieChart
                  series={[
                    {
                      data: chart.data,
                      innerRadius: 0,
                      outerRadius: 70,
                      paddingAngle: 1,
                      cornerRadius: 5,
                      startAngle: -90,
                      endAngle: 270,
                      highlightScope: { faded: 'global', highlighted: 'item' },
                      arcLabel: () => '',
                      cx: 75,
                      cy: 75,
                      arcLabelMinAngle: 45,
                      valueFormatter: () => '',
                    },
                  ]}
                  width={150}
                  height={150}
                  margin={{ top: 0, bottom: 0, left: 0, right: 0 }}
                  slotProps={{
                    legend: { hidden: true },
                  }}
                  sx={{
                    [`& .MuiPieArc-root:hover`]: {
                      filter: 'drop-shadow(0px 0px 6px rgba(0, 0, 0, 0.3))',
                      transform: 'scale(1.03)',
                      transition: 'all 0.2s ease-in-out',
                    },
                  }}
                />
              </Box>
            </Box>

            <Box 
              sx={{ 
                width: '60%',
                pl: 2,
                display: 'flex',
                alignItems: 'flex-start',
              }}
            >
              <Stack 
                spacing={1.5} 
                sx={{ 
                  width: '100%',
                  py: 1,
                }}
              >
                {chart.data.map((item) => {
                  const total = chart.data.reduce((acc, curr) => acc + curr.value, 0);
                  const percentage = Math.round((item.value / total) * 100);
                  
                  return (
                    <Stack 
                      key={item.id} 
                      direction="row" 
                      alignItems="center"
                      spacing={2}
                      sx={{
                        p: 1,
                        minHeight: 30,
                        borderRadius: 1,
                        cursor: 'pointer',
                        transition: 'all 0.2s',
                        '&:hover': {
                          bgcolor: (theme) => alpha(theme.palette.primary.main, 0.08),
                          transform: 'translateX(4px)',
                          boxShadow: '0 2px 8px rgba(0,0,0,0.08)',
                        },
                      }}
                    >
                      <Box
                        sx={{
                          width: 16,
                          height: 16,
                          borderRadius: '50%',
                          bgcolor: item.color,
                          flexShrink: 0,
                          display: 'flex',
                          alignItems: 'center',
                          boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
                        }}
                      />
                      <Typography 
                        variant="body1"
                        sx={{ 
                          color: 'text.primary',
                          lineHeight: 1.2,
                          flex: 1,
                          display: 'flex',
                          alignItems: 'center',
                          fontWeight: 500,
                        }}
                        noWrap
                      >
                        {`${item.label} (${item.value}) ${percentage}%`}
                      </Typography>
                    </Stack>
                  );
                })}
              </Stack>
            </Box>
          </Box>
        ))}
      </Box>
    );
  }

  return (
    <Grid container spacing={2} sx={{
      mb: 2,
      justifyContent: charts.length === 1 ? "center" : "flex-start",
    }}>
      {charts.map((chart, index) => (
        <Grid item xs={12} sm={6} key={index}>
          <Card
            sx={{
              p: 1,
              height: '100%',
              borderRadius: 2,
              border: (theme) => `1px solid ${theme.palette.divider}`,
              boxShadow: 'none',
              bgcolor: 'background.paper',
              display: 'flex',
              flexDirection: 'column',
              '&:hover': {
                bgcolor: (theme) => alpha(theme.palette.primary.main, 0.02),
                transition: 'background-color 0.3s',
              },
            }}
          >
            <Typography 
              variant="h6"
              sx={{ 
                mb: 1,
                color: (theme) => theme.palette.primary.main, 
                fontWeight: 600,
                pl: '13%',
              }}
            >
              {chart.title}
            </Typography>

            <Box sx={{ 
              display: 'flex', 
              flexDirection: 'row',
              alignItems: 'center',
              justifyContent: 'space-between',
              flex: 1,
            }}>
              <Box 
                sx={{ 
                  width: '40%', 
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center',
                }}
              >
                <Box 
                  sx={{ 
                    position: 'relative',
                    width: 90,
                    height: 90,
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                  }}
                >
                  <PieChart
                    series={[
                      {
                        data: chart.data,
                        innerRadius: 0,
                        outerRadius: 40,
                        paddingAngle: 1,
                        cornerRadius: 5,
                        startAngle: -90,
                        endAngle: 270,
                        highlightScope: { faded: 'global', highlighted: 'item' },
                        arcLabel: () => '',
                        cx: 45,
                        cy: 45,
                        arcLabelMinAngle: 45,
                        valueFormatter: () => '',
                      },
                    ]}
                    width={90}
                    height={90}
                    margin={{ top: 0, bottom: 0, left: 0, right: 0 }}
                    slotProps={{
                      legend: { hidden: true },
                    }}
                    sx={{
                      [`& .MuiPieArc-root:hover`]: {
                        filter: 'drop-shadow(0px 0px 6px rgba(0, 0, 0, 0.3))',
                        transform: 'scale(1.03)',
                        transition: 'all 0.2s ease-in-out',
                      },
                    }}
                  />
                </Box>
              </Box>

              <ScrollBox 
                sx={{ 
                  width: '60%',
                  pl: 1,
                  pr: 1,
                  display: 'flex',
                  alignItems: 'flex-start',
                  overflowY: 'auto',
                }}
              >
                <Stack 
                  spacing={0.75} 
                  sx={{ 
                    width: '100%',
                    py: 1,
                  }}
                >
                  {chart.data.map((item) => {
                    const total = chart.data.reduce((acc, curr) => acc + curr.value, 0);
                    const percentage = Math.round((item.value / total) * 100);
                    
                    return (
                      <Stack 
                        key={item.id} 
                        direction="row" 
                        alignItems="center"
                        spacing={1.5}
                        sx={{
                          p: 0.5,
                          minHeight: 24,
                          borderRadius: 1,
                          cursor: 'pointer',
                          transition: 'all 0.2s',
                          '&:hover': {
                            bgcolor: (theme) => alpha(theme.palette.primary.main, 0.08),
                            transform: 'translateX(4px)',
                            boxShadow: '0 2px 4px rgba(0,0,0,0.08)',
                          },
                        }}
                      >
                        <Box
                          sx={{
                            width: 12,
                            height: 12,
                            borderRadius: '50%',
                            bgcolor: item.color,
                            flexShrink: 0,
                            display: 'flex',
                            alignItems: 'center',
                            boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
                          }}
                        />
                        <Typography 
                          variant="body2"
                          sx={{ 
                            color: 'text.primary',
                            lineHeight: 1.2,
                            flex: 1,
                            display: 'flex',
                            alignItems: 'center',
                            fontWeight: 500,
                          }}
                          noWrap
                        >
                          {`${item.label} (${item.value}) ${percentage}%`}
                        </Typography>
                      </Stack>
                    );
                  })}
                </Stack>
              </ScrollBox>
            </Box>
          </Card>
        </Grid>
      ))}
    </Grid>
  );
}