import React from 'react';
import { IconButton, Stack, Tooltip } from '@mui/material';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faPenToSquare, faCopy } from '@fortawesome/free-solid-svg-icons';
import { useTheme } from '@mui/material/styles';
import { useBoolean } from '@/hooks';
import { AntSwitch } from './styles';
import CustomTooltip from '../tooltips/tooltip-custom';
import { ActionsIconButton } from '@/shared/theme/css';
// ----------------------------------------------------------------------
interface TooltipTitles {
  edit?: string;
  duplicate?: string;
  active?: string;
  inactive?: string;
}

interface AdditionalAction {
  icon: React.ReactNode;
  tooltip: string;
  onClick: () => void;
}

interface Props {
  handleEdit?: () => void;
  onDuplicate?: () => void;
  onToggle?: (checked: boolean) => void;
  isActive?: boolean;
  tooltipTitles: TooltipTitles;
  additionalActions?: AdditionalAction[];
}

export default function TableActionColumn({
  handleEdit,
  onDuplicate,
  onToggle,
  isActive = false,
  tooltipTitles,
  additionalActions = [],
}: Props) {
  const theme = useTheme();
  const active = useBoolean(isActive);

  return (
    <Stack direction="row" spacing={1} alignItems="center">
      {handleEdit ? (
        <CustomTooltip title={tooltipTitles.edit} arrow>
          <IconButton onClick={handleEdit} size="small" sx={ActionsIconButton}>
            <FontAwesomeIcon icon={faPenToSquare} size="sm" />
          </IconButton>
        </CustomTooltip>
      ) : (
        <></>
      )}

      {onDuplicate ? (
        <CustomTooltip title={tooltipTitles.duplicate} arrow>
          <IconButton onClick={onDuplicate} size="small" sx={ActionsIconButton}>
            <FontAwesomeIcon icon={faCopy} size="sm" />
          </IconButton>
        </CustomTooltip>
      ) : (
        <></>
      )}

      {additionalActions?.map((action, index) => (
        <CustomTooltip key={index} title={action.tooltip} arrow>
          <IconButton onClick={action.onClick} size="small" sx={ActionsIconButton}>
            {action.icon}
          </IconButton>
        </CustomTooltip>
      ))}

      {onToggle ? (
        <CustomTooltip title={active.value ? tooltipTitles.active : tooltipTitles.inactive} arrow>
          <AntSwitch
            theme={theme}
            checked={active.value}
            onChange={(event) => {
              active.onToggle();
              onToggle(event.target.checked);
            }}
            size="small"
          />
        </CustomTooltip>
      ): <></>}
    </Stack>
  );
}
