import React from 'react';
import { Box, Stack, IconButton } from '@mui/material';
import { faFileExport, faRotate } from '@fortawesome/free-solid-svg-icons';
import { iconButtonStyles } from '@/shared/theme/css';
import FontAwesome from '@/shared/components/fontawesome';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { IcircuitTrajet } from '@/shared/types/circuit';
import {
  _TRAJETS,
  DEFAULT_CIRCUIT_TRAJET_TABLE_HEAD,
  INITIAL_CIRCUIT_TRAJET_DATA,
} from '@/shared/_mock/_circuitTrajets';
import TableManager from '@/shared/components/table/table-manager';
import TrajetTableRow from './trajet-table-row';
import TransportSchedule from './trajet-details';
import { TabConfig, useTabsManager } from '@/hooks/use-tabs-manager';
import TabsCustom from '@/shared/components/tabs/tabs-custom';
import TrajetForm from '@/shared/sections/trajet/view/trajet-new-view';
import { ITrajet } from '@/shared/types/trajet';
import ExportButton from '@/shared/components/table/table-export-button';
import { TableType } from '@/shared/types/common';
import { ScheduleDTO, TripDTO } from '@/shared/types/amendment';

const circuitConfig: TableConfig<IcircuitTrajet> = {
  initialData: _TRAJETS,
  defaultTableHead: DEFAULT_CIRCUIT_TRAJET_TABLE_HEAD,
};
const trajetTabsConfig: TabConfig<IcircuitTrajet> = {
  type: 'trajet',
  listTitle: 'Liste des trajets',
  newItemTitle: 'Nouveau trajet',
  initialData: INITIAL_CIRCUIT_TRAJET_DATA,
  getItemCode: (chauffeur) => chauffeur.numeroTrajet,
};
interface CircuittrajetsFormProps {
  useTabs?: boolean;
  schedules?: ScheduleDTO[];
  trips?: TripDTO[];
}
export default function CircuittrajetsForm({ useTabs, schedules, trips }: CircuittrajetsFormProps) {
  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    dataToExport,
    handleResetColumns,
    handleColumnsChange,
    handleSave,
  } = useTableManager<IcircuitTrajet>(circuitConfig);
  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
  } = useTabsManager<IcircuitTrajet>(trajetTabsConfig);
  const activeTabData = React.useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab);
  }, [tabs, activeTab]);

  const renderRow = (row: IcircuitTrajet) => (
    <TrajetTableRow
      dense={table.dense}
      key={row.id}
      row={row}
      handleEditContract={() => void 0}
      handleDetailsContract={useTabs ? handleView : () => void 0}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead}
    />
  );

  return (
    <Box >
      <Stack
        sx={{
          visibility: activeTabData?.id === 'list' ? 'visible' : 'hidden',
        }}
        direction="row"
        alignItems="center"
        justifyContent="flex-end"
        gap="8px"
        pb={2}
      >
        <CustomTooltip title="Refresh" arrow>
          <IconButton onClick={handleResetFilters} sx={iconButtonStyles}>
            <FontAwesome icon={faRotate} width={16} />
          </IconButton>
        </CustomTooltip>
        <ExportButton
          tooltipTitle={"Exporter"}
          type={TableType.Trajet}
          filteredData={dataToExport}
          tableHead={tableHead}
        />
      </Stack>
      {useTabs ? <TabsCustom
        type="trajet"
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      /> : <></>}

      {activeTab === 'list' ? (
        <>
          <TableManager
            filteredData={filteredData}
            table={table}
            tableHead={tableHead}
            notFound={notFound}
            filters={filters}
            onFilterChange={handleFilterChange}
            renderRow={renderRow}
            sx={{ backgroundColor: 'primary.lighter' }}
            isTrajetTable={true}
            isWithoutTabs
          />
          <TransportSchedule schedules={schedules} trips={trips} />
        </>
      ) : (
        <TrajetForm
          trajet={activeTabData?.content as ITrajet}
          mode={activeTabData?.mode || 'view'}
          onClose={() => handleCancel(activeTab)}
          hasBorder
          tableHead={tableHead}
        />
      )}
    </Box>
  );
}
