'use client';

import React, { useEffect } from 'react';
import { useFormik } from 'formik';
import {
  Grid,
} from '@mui/material';
import FormField from '@/shared/components/form/form-field';
import { FormFieldType, ModeType, TableColumn, TableType } from '@/shared/types/common';
import EditExportButtons from '@/shared/components/form/edit-export-buttons';
import ActionButtons from '@/shared/components/form/buttons-action';
import FormContainer from '@/layouts/form/form-container';
import { IVehicleAffectation } from '@/shared/types/flottes/vehicle-affectation';
import { affectationformFields } from '../utils/form-fields-affectation';
import { affectationSchema } from '../utils/affectation-schema';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

interface VehicleFormProps {
  affectation: IVehicleAffectation;
  mode: ModeType;
  onSave: (updatedVehicle: IVehicleAffectation) => void;
  onEdit: (updatedVehicle: IVehicleAffectation) => void;
  onClose: () => void;
  tableHead: TableColumn[];
  setHasUnsavedChanges?: (value: boolean) => void
  updateTabContent?: (tabId: string, newContent: IVehicleAffectation) => void
  tabId?: string
}

export default function AffectationForm({
  affectation,
  mode,
  onSave,
  onClose,
  onEdit,
  tableHead,
  setHasUnsavedChanges,
  updateTabContent,
  tabId,
}: VehicleFormProps) {
  const isReadOnly = mode === 'view';
  const isEditMode = mode === 'edit';

  const formik = useFormik<IVehicleAffectation>({
    initialValues: affectation,
    validationSchema: affectationSchema,
    onSubmit: (values) => {
      onSave(values);
      if (setHasUnsavedChanges) setHasUnsavedChanges(false)
      onClose();
    },
  });

  useEffect(() => {
    if (updateTabContent && tabId && (mode !== "view")) {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const renderField = (field: FormFieldType<IVehicleAffectation>) => {
    const alwaysReadOnly = ["marque", "modele", "numeroImmatriculation"];
    const fieldIsReadOnly = alwaysReadOnly.includes(field.name) && isEditMode ? true : isReadOnly;
    
    return (
      <FormField
        field={field}
        value={formik.values[field.name as keyof IVehicleAffectation]}
        onChange={(name, value) => formik.setFieldValue(name, value)}
        error={
          formik.touched[field.name as keyof IVehicleAffectation]
            ? (formik.errors[field.name as keyof IVehicleAffectation] as string | undefined)
            : undefined
        }
        onBlur={formik.handleBlur}
        isReadOnly={fieldIsReadOnly}
      />
    );
  };

  return (
    <FormContainer titre="Fiche d'affectation du véhicule">
 <ConditionalComponent isValid={isReadOnly}>
        <EditExportButtons
          onEdit={() => onEdit(affectation)}
          onExport={() => void 0}
          tooltipTitle={TableType.Affectation}
          dataRow={affectation}
          tableHead={tableHead}
        />
      </ConditionalComponent>
      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={5}>
          {affectationformFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}

        
<ConditionalComponent isValid={!isReadOnly}>
            <Grid item xs={12}>
              <ActionButtons onSave={() => formik.submitForm()} onCancel={onClose} mode={mode} />
            </Grid>
          </ConditionalComponent>
        </Grid>
      </form>
    </FormContainer>
  );
}
