import { TableCell, IconButton, Stack, Checkbox, Box, useTheme } from '@mui/material';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import FontAwesome from '@/shared/components/fontawesome';
import { faCopy, faEye, faPenToSquare, faTrash } from '@fortawesome/free-solid-svg-icons';
import { IGeolocItem, GeolocStatus } from '@/shared/types/geoloc';
import { formatDateTime } from '@/utils/format-time';
import SvgColor from '@/shared/components/svg-color';
import { ActionsIconButton } from '@/shared/theme/css';
import type { TableColumn } from '@/shared/types/common';
import { 
  GeolocStyledRow, 
  getGeolocStatusColor, 
  getGeolocStatusLabel, 
  GeolocStatusChip,
} from '@/shared/sections/geolocalisation/styles';

interface VehiclePositionsMap {
  [key: string]: {
    vehicleType: string;
    position: { lat: number; lng: number };
    rotation: number;
  };
}

type Props = {
  row: IGeolocItem;
  dense: boolean;
  selected: boolean;
  columns: TableColumn[];
  onSelectRow: () => void;
  handleDetails: (item: IGeolocItem) => void;
  vehiclePositions?: VehiclePositionsMap;
};

const TOOLTIP_TITLES = {
  view: 'Voir les détails',
  edit: 'Modifier trajet',
  copy: 'Dupliquer trajet',
  delete: 'Supprimer trajet',
};

export default function GeolocTableRow({
  row,
  dense,
  selected,
  columns,
  onSelectRow,
  handleDetails,
  vehiclePositions = {}
}: Props) {
  const theme = useTheme();

  return (
    <GeolocStyledRow hover 
    selected={selected}
    style={{ cursor: 'pointer' }}  onDoubleClick={() => handleDetails(row)}>
    

      {columns.map((column) => {
        const { id } = column;
        
        if (id === 'status') {
          return (
            <TableCell key={id}  sx={{ height: '40px' }}>
              <SvgColor
          src="/assets/icons/ic_border.svg"
          sx={{
            position: 'absolute',
            top: 0,
            left: -4,
            height: '100%',
            color: getGeolocStatusColor(row.status),
          }}
        />
              <GeolocStatusChip 
                theme={theme}
                label={getGeolocStatusLabel(row.status)}
                status={getGeolocStatusLabel(row.status)}
              />
            </TableCell>
          );
        }
        
        if (id === 'dateDepart' || id === 'dateArrivee') {
          return (
            <TableCell key={id} align="left">
              {formatDateTime(row[id as keyof IGeolocItem] as string)}
            </TableCell>
          );
        }
        
        if (id === 'distance') {
          return (
            <TableCell key={id} align="left">
              {`${row.distance} km`}
            </TableCell>
          );
        }
        
        if (id === 'nomComplet') {
          return (
            <TableCell key={id} align="left">
              {`${row.prenomChauffeur} ${row.nomChauffeur}`}
            </TableCell>
          );
        }
        
        if (id === '') {
          return null;
        }
        
        return (
          <TableCell 
            key={id} 
            align="left"
            sx={{ whiteSpace: 'nowrap', py: dense ? 0 : 1, px: 1,mx:0.5,minWidth: column.width || 'auto' }}
          >
            {(id in row) ? String(row[id as keyof IGeolocItem]) : ''}
          </TableCell>
        );
      })}

      <TableCell sx={{ whiteSpace: 'nowrap', px: 1, alignItems: 'center' }}>
        <Stack direction="row" spacing={0.5}>
          <CustomTooltip title={TOOLTIP_TITLES.view} arrow>
            <IconButton 
               onClick={(e) => {
                e.stopPropagation(); 
                handleDetails(row);
              }} 
              size="small" 
              sx={ActionsIconButton}
            >
              <FontAwesome icon={faEye} width={14} />
            </IconButton>
          </CustomTooltip>
          
        </Stack>
      </TableCell>
    </GeolocStyledRow>
  );
}