"use client"

import { useMemo } from "react"
import { Box, Stack } from "@mui/material"
import { HEADER } from "@/layouts/config-layout"
import TableControlBar from "@/shared/components/table/table-control-bar"
import { type TableConfig, useTableManager } from "@/hooks/use-table-manager"
import type { IDemande } from "@/shared/types/demande"
import { _demandes, DEFAULT_DEMANDE_TABLE_HEAD, INITIAL_DEMANDE_DATA } from "@/shared/_mock/_demandes"
import { type TabConfig, useTabsManager } from "@/hooks/use-tabs-manager"
import { TableType } from "@/shared/types/common"
import TabsCustom from "@/shared/components/tabs/tabs-custom"
import TableManager from "@/shared/components/table/table-manager"
import DemandeTableRow from "./demande-table-row"
import DemandeForm from "./demande-new-view"

const demandeConfig: TableConfig<IDemande> = {
  initialData: _demandes,
  defaultTableHead: DEFAULT_DEMANDE_TABLE_HEAD,
}

const demandeTabsConfig: TabConfig<IDemande> = {
  type: TableType.Demande,
  listTitle: "Liste demandes",
  newItemTitle: "Nouvelle demande",
  initialData: INITIAL_DEMANDE_DATA,
  getItemCode: (demande) => demande.intitule,
}

export default function DemandeListView() {
  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    handleCopy,
  } = useTabsManager<IDemande>(demandeTabsConfig)

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    handleSave,
    handleStatusChange,
  } = useTableManager<IDemande>(demandeConfig)

  const activeTabData = useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab)
  }, [tabs, activeTab])

  const renderRow = (row: IDemande) => (
    <DemandeTableRow
      key={row.id}
      dense={table.dense}
      row={row}
      handleEdit={handleEdit}
      handleCopy={handleCopy}
      handleDetails={handleView}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead.slice(0, -1)} 
      onStatusChange={handleStatusChange}
      onPriorityChange={handleStatusChange}
        />
  )

  const isList = activeTab === "list"

  return (
    <Box
      sx={{
        display: "flex",
        flexDirection: "column",
        height: `calc(100vh - ${HEADER.H_DESKTOP + 8}px)`,
      }}
    >
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <TableControlBar
          type={TableType.Demande}
          handleTabAdd={handleTabAdd}
          activeTab={activeTab}
          onResetFilters={handleResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={filteredData}
        />
      </Stack>
      <TabsCustom
        type={TableType.Demande}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      {isList ? (
        <>
          <TableManager
            filteredData={filteredData}
            table={table}
            tableHead={tableHead}
            notFound={notFound}
            filters={filters}
            onFilterChange={handleFilterChange}
            renderRow={renderRow}
          />
        </>
      ) : (
        <DemandeForm
          demande={activeTabData?.content as IDemande}
          mode={activeTabData?.mode || "view"}
          onSave={handleSave}
          onClose={() => handleCancel(activeTab)}
          onEdit={handleEdit}
        />
      )}
    </Box>
  )
}

