import { AddressData } from "./common";

export interface ISector {
  id: string;
  code: string;
  name: string;
  description: string;
  createdAt: string;
  updatedAt: string;
  adresse: AddressData;
  agenciesNbr: number;
  agencies?: IAgency[];
}

export interface IAgency {
  id: string;
  name: string;
  description: string;
  adresse: AddressData;
  createdAt: string;
  updatedAt: string;
  departmentsNbr: number;
  sector?: ISector;
  departments?: IDepartment[];
}

export interface IDepartment {
  id: string;
  name: string;
  adresse: AddressData;
  description: string;
  createdAt: string;
  updatedAt: string;
  lotsNbr: number;
  agency?: IAgency;
  lots?: ILot[];
}

export interface ILot {
  id: string;
  name: string;
  lotNumber: string;
  description: string;
  createdAt: string;
  updatedAt: string;
  department?: IDepartment[];
  sourceLots?: ILot[];
}

export type SectorNames = {
  id: number;
  name: string;
};

export type AgencyNames = {
  id: number;
  name: string;
};

export type Agencies = {
};

export type DepartmentNames = {
  id: number;
  name: string;
};

export type LotNumbers = {
  id: number;
  number: string;
};

interface BaseEntity {
  id?: number;
  createdAt?: string;
  updatedAt?: string;
}

export interface Address extends BaseEntity {
  street?: string;
  zipCode?: string;
  city?: string;
  country?: string;
  additionalAddress?: string;
  latitude: number;
  longitude: number;
  addressType?: AddressType;
}

export enum AddressType {
  COLLABORATOR = 'COLLABORATOR',
  CLIENT = 'CLIENT',
  ESTABLISHMENT = 'ESTABLISHMENT',
  REPRESENTATIVE = 'REPRESENTATIVE',
  BANQUE = 'BANQUE',
  DRIVER = 'DRIVER'
}

export interface MapAddresse {
  address: string,
  city: string,
  country: string,
  formattedAddress: string,
  lat: number,
  lng: number,
  placeId: string,
  postalCode: string
}

export interface SectorList {
  id: string;
  name: string;
  address: string;
  createdAt: string;
  active: boolean;
}

export interface SectorPageFilterParams {
  name?: string;
  address?: string;
  createdAt?: string;
  page?: number;
  size?: number;
  sortBy?: string;
  sortDirection?: 'asc' | 'desc';
}

export interface SectorRequest {
  id: string;
  name: string;
  description: string;
  mapAddresse: MapAddresse;
}

export interface Sector extends BaseEntity {
  name: string;
  description: string;
  address: Address;
  agenciesNbr: number;
  agencies: AgencyChildren[];
}

export const mapSectorToSectorRequest = (sector: Sector): SectorRequest => {
  return {
    id: sector.id ? sector.id.toString() : '',
    name: sector.name,
    description: sector.description,
    mapAddresse: {
      address: `${sector.address.street}, ${sector.address.zipCode} ${sector.address.city}, ${sector.address.country}`,
      lat: sector.address.latitude,
      lng: sector.address.longitude,
      city: sector.address.city || '',
      postalCode: sector.address.zipCode || '',
      country: sector.address.country || '',
      formattedAddress: `${sector.address.street}, ${sector.address.zipCode} ${sector.address.city}, ${sector.address.country}`,
      placeId: ''
    },
  };
};

export const mapAddressToMapAddresse = (address: Address): MapAddresse => {
  return {
    address: `${address.street}, ${address.zipCode} ${address.city}, ${address.country}`,
    lat: address.latitude,
    lng: address.longitude,
    city: address.city || '',
    postalCode: address.zipCode || '',
    country: address.country || '',
    formattedAddress: `${address.street}, ${address.zipCode} ${address.city}, ${address.country}`,
    placeId: ''
  };
};

export interface AgencyList {
  id: string;
  name?: string;
  address?: string;
  description?: string;
  createdAt?: string;
  active: boolean;
}

export interface AgencyPageFilterParams {
  name?: string;
  address?: string;
  description?: string;
  createdAt?: string;
  page?: number;
  size?: number;
  sortBy?: string;
  sortDirection?: 'asc' | 'desc';
}

export interface AgencyRequest {
  id: string;
  name: string;
  description: string;
  mapAddresse: MapAddresse;
  sectorId: string;
}

export interface Agency extends BaseEntity {
  name: string;
  description: string;
  address: Address;
  sectorId: string;
  departmentsNbr: number;
  departments: DepartmentChildren[];
}

export const mapAgencyToAgencyRequest = (agency: Agency): AgencyRequest => {
  return {
    id: agency.id ? agency.id.toString() : '',
    name: agency.name,
    description: agency.description,
    mapAddresse: mapAddressToMapAddresse(agency.address),
    sectorId: agency.sectorId,
  };
};


export interface DepartmentList {
  id: string;
  name?: string;
  address?: string;
  description?: string;
  createdAt?: string;
  active: boolean;
}

export interface DepartmentPageFilterParams {
  name?: string;
  address?: string;
  description?: string;
  createdAt?: string;
  page?: number;
  size?: number;
  sortBy?: string;
  sortDirection?: 'asc' | 'desc';
}

export interface DepartmentRequest {
  id: string;
  name: string;
  description: string;
  mapAddresse: MapAddresse;
  agencyId: string;
}

export interface Department extends BaseEntity {
  name: string;
  description: string;
  address: Address;
  agencyId: string;
  lotsNbr: number;
  lots: LotChildren[];
}

export const mapDepartmentToDepartmentRequest = (department: Department): DepartmentRequest => {
  return {
    id: department.id ? department.id.toString() : '',
    name: department.name,
    description: department.description,
    mapAddresse: mapAddressToMapAddresse(department.address),
    agencyId: department.agencyId,
  };
};

export interface LotList {
  id: string;
  number?: string;
  description?: string;
  createdAt?: string;
  active: boolean;
}

export interface LotPageFilterParams {
  number?: string;
  description?: string;
  createdAt?: string;
  page?: number;
  size?: number;
  sortBy?: string;
  sortDirection?: 'asc' | 'desc';
}

export interface LotRequest {
  id: string;
  number: string;
  description: string;
  departmentsIds: string[];
}

export interface Lot extends BaseEntity {
  number: string;
  description: string;
  departmentsIds: string[];
}

export const mapLotToLotRequest = (lot: Lot): LotRequest => {
  return {
    id: lot.id ? lot.id.toString() : '',
    number: lot.number,
    description: lot.description,
    departmentsIds: lot.departmentsIds,
  };
};


export interface SectorWithAgencies extends BaseEntity {
  name?: string;
  description?: string;
  address?: Address;
  agenciesNbr: number;
  active: boolean;
  agencies: AgencyChildren[];
}

export interface AgencyWithDepartments extends BaseEntity {
  name?: string;
  description?: string;
  address?: Address;
  departmentsNbr: number;
  active: boolean;
  departments: DepartmentChildren[];
}

export interface DepartmentWithLots extends BaseEntity {
  name?: string;
  description?: string;
  address?: Address;
  lotsNbr: number;
  active: boolean;
  lots: LotChildren[];
}

export interface AgencyChildren extends BaseEntity {
  name?: string;
  description?: string;
  address?: Address;
  departmentsNbr: number;
  sectorId?: number;
  active: boolean;
  archived: boolean;
}

export interface DepartmentChildren extends BaseEntity {
  name?: string;
  description?: string;
  address?: Address;
  lotsNbr: number;
  agencyId?: number;
  active: boolean;
  archived: boolean;
}

export interface LotChildren extends BaseEntity {
  number?: string;
  description?: string;
  address?: Address;
  active: boolean;
  archived: boolean;
}
