"use client"

import { useState, useCallback, useContext } from "react"
import { Box, Button, Stack, Typography } from "@mui/material"
import { HEADER } from "@/layouts/config-layout"
import TableControlBar from "@/shared/components/table/table-control-bar"
import { type TableConfig, useTableManager } from "@/hooks/use-table-manager"
import { TableType } from "@/shared/types/common"
import TableManager from "@/shared/components/table/table-manager"
import { Add as AddIcon, ImportExport as ImportIcon, History as HistoryIcon } from "@mui/icons-material"
import type { IEmailTemplate, IEmailTemplateFilter } from "@/shared/types/email-template"
import { _emailTemplates, DEFAULT_EMAIL_TEMPLATE_TABLE_HEAD } from "@/shared/_mock/_email-templates"
import EmailTemplateForm from "../components/email-template-form"
import EmailTemplateImport from "../components/email-template-import"
import EmailTemplateExport from "../components/email-template-export"
import EmailTemplateArchive from "../components/email-template-archive"
import { EmailTabContext } from "../page"
import { EmailManagementTab } from "@/shared/types/email"
import EmailTemplateTableRow from "./email-template-table-row"

// Configuration pour le tableau des modèles d'email
const emailTemplateConfig: TableConfig<IEmailTemplate> = {
  initialData: _emailTemplates.filter((template) => !template.isArchive),
  defaultTableHead: DEFAULT_EMAIL_TEMPLATE_TABLE_HEAD,
}

export default function EmailTemplatesView() {
  // const router = useRouter()
  const [selectedTemplate, setSelectedTemplate] = useState<IEmailTemplate | null>(null)
  const [isAddModalOpen, setIsAddModalOpen] = useState(false)
  const [isEditModalOpen, setIsEditModalOpen] = useState(false)
  const [isImportModalOpen, setIsImportModalOpen] = useState(false)
  const [isExportModalOpen, setIsExportModalOpen] = useState(false)
  const [isArchiveModalOpen, setIsArchiveModalOpen] = useState(false)

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
  } = useTableManager<IEmailTemplate>(emailTemplateConfig)

  // Gérer les filtres personnalisés
  const handleCustomFilterChange = useCallback(
    (newFilters: IEmailTemplateFilter) => {
      handleResetFilters()

      setTimeout(() => {
        if (newFilters.titre) {
          handleFilterChange("titre", newFilters.titre, "contains")
        }

        if (newFilters.categorie && newFilters.categorie.length > 0) {
          newFilters.categorie.forEach((cat) => {
            handleFilterChange("categorie", cat, "equals")
          })
        }

        if (newFilters.dateDebut) {
          const formattedDate = newFilters.dateDebut.toISOString().split("T")[0]
          handleFilterChange("dateCreation", formattedDate, "gte")
        }

        if (newFilters.dateFin) {
          const formattedDate = newFilters.dateFin.toISOString().split("T")[0]
          handleFilterChange("dateCreation", formattedDate, "lte")
        }
      }, 0)
    },
    [handleResetFilters, handleFilterChange],
  )

  // Gérer l'ajout d'un modèle
  const handleAddTemplate = (template: IEmailTemplate) => {
    console.log("Ajout d'un nouveau modèle:", template)
    // Ici, vous ajouteriez le modèle à votre base de données
    setIsAddModalOpen(false)
  }

  // Gérer la modification d'un modèle
  const handleEditTemplate = (template: IEmailTemplate) => {
    console.log("Modification du modèle:", template)
    setSelectedTemplate(template)
    setIsEditModalOpen(true)
  }

  // Gérer l'importation d'un modèle
  const handleImportTemplate = (template: IEmailTemplate) => {
    console.log("Importation d'un modèle:", template)
    // Ici, vous ajouteriez le modèle importé à votre base de données
    setIsImportModalOpen(false)
  }

  // Gérer l'archivage d'un modèle
  const handleArchiveTemplate = (template: IEmailTemplate) => {
    console.log("Archivage du modèle avec l'ID:", template.id)
    setSelectedTemplate(template)
    setIsArchiveModalOpen(true)
  }

  // Gérer l'exportation d'un modèle
  const handleExportTemplate = (template: IEmailTemplate) => {
    console.log("Exportation du modèle:", template)
    setSelectedTemplate(template)
    setIsExportModalOpen(true)
  }

  // Gérer la sauvegarde des modifications
  const handleSaveTemplate = (template: IEmailTemplate) => {
    console.log("Sauvegarde du modèle:", template)
    setIsEditModalOpen(false)
  }

  // Gérer la confirmation d'archivage
  const handleConfirmArchive = (id: string) => {
    console.log("Confirmation d'archivage du modèle:", id)
    setIsArchiveModalOpen(false)
  }

  // Naviguer vers la vue historique
  const { setActiveTab } = useContext(EmailTabContext)
  const handleViewHistory = () => {
    setActiveTab(EmailManagementTab.History)
  }

  // Rendu d'une ligne du tableau
  const renderTemplateRow = (row: IEmailTemplate) => (
    <EmailTemplateTableRow
      key={row.id}
      dense={table.dense}
      row={row}
      handleEdit={handleEditTemplate}
      handleDetails={handleEditTemplate}
      handleArchive={handleArchiveTemplate}
      handleExport={handleExportTemplate}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead}
    />
  )

  return (
    <>
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="space-between" sx={{ mb: 2 }}>
        
        <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
          <Button
            variant="outlined"
            startIcon={<HistoryIcon />}
            onClick={handleViewHistory}
            sx={{ textTransform: "none" }}
          >
            Voir Historique
          </Button>

          <Button
            variant="outlined"
            startIcon={<ImportIcon />}
            onClick={() => setIsImportModalOpen(true)}
            sx={{ textTransform: "none" }}
          >
            Importer un modèle
          </Button>
          </Box>
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>

          <TableControlBar
            type={TableType.Email}
            activeTab="list"
            onResetFilters={handleResetFilters}
            onResetColumns={handleResetColumns}
            onColumnsChange={handleColumnsChange}
            initialColumns={tableHead.slice(0, -1)}
            onFilters={handleFilterChange}
            handleTabAdd={() => setIsAddModalOpen(true)}
            filteredData={filteredData}
            buttonAddTitle="Ajouter un modèle"
            />
        </Box>
      </Stack>

        <TableManager
          filteredData={filteredData}
          table={table}
          tableHead={[...tableHead, { id: "actions", label: "Actions", type: "text", width: 80 }]}
          notFound={notFound}
          filters={filters}
          onFilterChange={handleFilterChange}
          renderRow={renderTemplateRow}
        />

      {/* Modales */}
      <EmailTemplateForm
        open={isAddModalOpen}
        template={null}
        mode="add"
        onClose={() => setIsAddModalOpen(false)}
        onSave={handleAddTemplate}
      />

      <EmailTemplateForm
        open={isEditModalOpen}
        template={selectedTemplate}
        mode="edit"
        onClose={() => setIsEditModalOpen(false)}
        onSave={handleSaveTemplate}
      />

      <EmailTemplateImport
        open={isImportModalOpen}
        onClose={() => setIsImportModalOpen(false)}
        onImport={handleImportTemplate}
      />

      <EmailTemplateExport
        open={isExportModalOpen}
        template={selectedTemplate}
        onClose={() => setIsExportModalOpen(false)}
      />

      <EmailTemplateArchive
        open={isArchiveModalOpen}
        template={selectedTemplate}
        onClose={() => setIsArchiveModalOpen(false)}
        onArchive={handleConfirmArchive}
      />
    </>
  )
}
