"use client"

import { useMemo } from "react"
import { Stack, Typography } from "@mui/material"
import { HEADER } from "@/layouts/config-layout"
import TableControlBar from "@/shared/components/table/table-control-bar"
import type { AgencyList } from "@/shared/types/infrastructure"
import { type TabConfig, useTabsManager } from "@/hooks/use-tabs-manager"
import { TableType } from "@/shared/types/common"
import TabsCustom from "@/shared/components/tabs/tabs-custom"
import TableManager from "@/shared/components/table/table-manager"
import AgencyTableRow from "./agency-row"
import AgencyForm from "./agency-form"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import { useAgencyTable } from "../hooks/use-agency-table"
import { UnsavedChangesDialog } from "@/shared/components/dialog/UnsavedChangesDialog"


export const INITIAL_AGENCE_DATA: AgencyList = {
  id: '',
  name: '',
  address: '',
  description: '',
  createdAt: new Date().toISOString().split('T')[0],
  active: true,
};

const agencyTabsConfig: TabConfig<AgencyList> = {
  type: TableType.Agence,
  listTitle: "Liste agences",
  newItemTitle: "Nouvelle agence",
  initialData: INITIAL_AGENCE_DATA,
  getItemCode: (agency) => agency.id,
}

export default function AgencyListView() {

  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    handleCopy,
    showConfirmDialog,
    setShowConfirmDialog,
    handleConfirmDialogAction,
    updateTabContent,
  } = useTabsManager<AgencyList>(agencyTabsConfig)

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    onPageChange,
    page,
    onRowsPerPageChange,
    rowsPerPage,
    totalAgencies,
    sortBy,
    sortDirection,
  } = useAgencyTable()
  
  const activeTabData = useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab)
  }, [tabs, activeTab])

  const renderRow = (row: AgencyList) => (
    <AgencyTableRow
      key={row.id}
      dense={table.dense}
      row={row}
      handleEdit={handleEdit}
      handleCopy={handleCopy}
      handleDetails={handleView}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead.slice(0, -1)}
      handleOpenConsult={handleView}
    />
  )

  const isList = activeTab === "list"

  return (
    <>
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <TableControlBar
          type={TableType.Agence}
          handleTabAdd={handleTabAdd}
          activeTab={activeTab}
          onResetFilters={handleResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={filteredData}
        />
      </Stack>

      <ConditionalComponent isValid={isList}>
        <Typography
          variant="h5"
          sx={{
            mb: 2,
            color: "primary.main",
            fontWeight: "bold",
          }}
        >
          Liste Agences
        </Typography>
      </ConditionalComponent>

      <TabsCustom
        type={TableType.Agence}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      <ConditionalComponent isValid={isList}
        defaultComponent={
          <AgencyForm
            agency={activeTabData?.content as AgencyList}
            onClose={(force) => handleCancel(activeTab, force)}
            mode={activeTabData?.mode || "add"}
            tableHead={tableHead}
            updateTabContent={updateTabContent}
            tabId={activeTab}
          />
        }
      >
        <>
          <TableManager
            filteredData={filteredData}
            table={table}
            tableHead={tableHead}
            notFound={notFound}
            filters={filters}
            onFilterChange={handleFilterChange}
            renderRow={renderRow}
            fixedHeight={`calc(100vh - ${HEADER.H_DESKTOP + 195}px)`}
            onPageChange={onPageChange}
            onRowsPerPageChange={onRowsPerPageChange}
            rowsPerPage={rowsPerPage || 20}
            page={page || 0}
            count={totalAgencies}
            sortBy={sortBy || 'name'}
            sortDirection={sortDirection || 'desc'}
          />
        </>
      </ConditionalComponent>
      <UnsavedChangesDialog
        open={showConfirmDialog}
        onClose={() => setShowConfirmDialog(false)}
        onConfirm={() => handleConfirmDialogAction()}
        onCancel={() => setShowConfirmDialog(false)}
      />
    </>
  )
}
