import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { useTheme } from '@mui/material/styles';
import SvgColor from '@/shared/components/svg-color';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import { IIncident } from '@/shared/types/flottes/incident';
import { ContratStyledRow, getStatusColor, StatusChip } from '@/shared/sections/contract/styles';
import { _SINISTRE_OPTION_STATUT } from '@/shared/_mock/flottes/_Incidents';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

type Props = {
  handleEditIncident: (incident: IIncident) => void;
  handleDetailsIncident: (incident: IIncident) => void;
  handleCopyIncident: (incident: IIncident) => void;
  selected: boolean;
  row: IIncident;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: IIncident, newStatus: string, field: keyof IIncident) => void;
};

const TOOLTIP_TITLES = {
  edit: 'Modifier sinistre',
  duplicate: 'Copier sinistre',
  active: 'Sinistre actif',
  inactive: 'Sinistre inactif',
};

export default function IncidentTableRow({
  row,
  handleEditIncident,
  handleDetailsIncident,
  handleCopyIncident,
  selected,
  onSelectRow,
  columns,
  onStatusChange
}: Props) {

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'dateSinistre':
        return formatDate(value, dateFormat.isoDate);
      case 'statut':
        return (
          <StatusMenu
            value={value}
            options={_SINISTRE_OPTION_STATUT}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'statut' as keyof IIncident)}
            chipComponent={(props) => <StatusChip status={props.value} {...props} />}
            sx={{ width: "180px" }}
          />
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>

      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetailsIncident(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'status' ? 0 : 'inherit', ...(index === 0 && {position: "relative"}), }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: "absolute", top: 0, left: -4, height: "100%", color: getStatusColor(row.typeSinistre) }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IIncident])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditIncident(row)}
          onDuplicate={() => handleCopyIncident(row)}
          onToggle={(checked) => void 0}
          isActive={row.isActive}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}