"use client"

import { Grid } from "@mui/material"
import { useFormik } from "formik"
import FormField from "@/shared/components/form/form-field"
import type { FormFieldType, ModeType } from "@/shared/types/common"
import EditExportButtons from "@/shared/components/form/edit-export-buttons"
import ActionButtons from "@/shared/components/form/buttons-action"
import FormContainer from "@/layouts/form/form-container"
import { ISupplierType } from "@/shared/types/flottes/vehicule"
import { supplierTypeFormFields } from "../utils/form-fields-supplier-type"
import { supplierTypeSchema } from "../utils/supplier-type-schema"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import { useEffect } from "react"

interface SupplierTypeFormProps {
  supplierType: ISupplierType
  mode: ModeType
  onSave: (updatedSupplierType: ISupplierType) => void
  onEdit: (updatedSupplierType: ISupplierType) => void
  onClose: () => void
  setHasUnsavedChanges?: (value: boolean) => void
  updateTabContent?: (tabId: string, newContent: ISupplierType) => void
  tabId?: string
}

export default function SupplierTypeForm({ supplierType, mode, onSave, onClose, onEdit, setHasUnsavedChanges, updateTabContent, tabId }: SupplierTypeFormProps) {
  const isReadOnly = mode === "view"

  const formik = useFormik<ISupplierType>({
    initialValues: {
      ...supplierType,
    },
    validationSchema: supplierTypeSchema,
    onSubmit: (values) => {
      onSave(values)
      if (setHasUnsavedChanges) setHasUnsavedChanges(false)
      onClose()
    },
  })

  useEffect(() => {
    if (updateTabContent && tabId && (mode !== "view")) {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const renderField = (field: FormFieldType<ISupplierType>) => (
    <FormField
      field={field}
      value={formik.values[field.name as keyof ISupplierType]}
      onChange={(name, value) => formik.setFieldValue(name, value)}
      error={
        Boolean(formik.touched[field.name as keyof ISupplierType]) &&
        formik.errors[field.name as keyof ISupplierType]
          ? String(formik.errors[field.name as keyof ISupplierType])
          : undefined
      }
      onBlur={formik.handleBlur}
      isReadOnly={isReadOnly}
    />
  )

  return (
    <FormContainer titre="Type de fournisseur">
      <ConditionalComponent isValid={isReadOnly}>
        <EditExportButtons
          onEdit={() => onEdit(supplierType)}
          tooltipTitle="le type de fournisseur"
        />
      </ConditionalComponent>

      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={5}>
          {supplierTypeFormFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}
          <ConditionalComponent isValid={!isReadOnly}>
            <Grid item xs={12}>
              <ActionButtons onSave={formik.submitForm} onCancel={onClose} mode={mode} />
            </Grid>
          </ConditionalComponent>
        </Grid>
      </form>
    </FormContainer>
  )
} 