import FontAwesome from '@/shared/components/fontawesome';
import { pxToRem } from '@/shared/theme/typography';
import { faWrench, faCar, faUser, faClock, faCircleCheck, faCalendar, faHourglassHalf } from '@fortawesome/free-solid-svg-icons';
import { Box, Popover, Typography } from '@mui/material';
import { getBgColor } from '../utils/schedule-utils';
import { format } from 'date-fns';
import { fr } from 'date-fns/locale';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

// Add this helper function to get the status label (same as in intervention-table.tsx)
function getStatusLabel(status: string) {
  switch(status) {
    case 'planned': return 'Planifié';
    case 'in_progress': return 'En cours';
    case 'completed': return 'Terminé';
    default: return status;
  }
}

interface InterventionDetailsPopoverProps {
  anchorEl: HTMLElement | null;
  onClose: () => void;
  slot: any;
}

export function InterventionDetailsPopover({ anchorEl, onClose, slot }: InterventionDetailsPopoverProps) {
  const bgColor = getBgColor(slot.color);
  
  return (
    <Popover
      open={Boolean(anchorEl)}
      anchorEl={anchorEl}
      onClose={onClose}
      disableRestoreFocus
      anchorOrigin={{
        vertical: 'center',
        horizontal: 'center',
      }}
      transformOrigin={{
        vertical: 'top',
        horizontal: 'center',
      }}
      sx={{
        pointerEvents: 'none',
        '& .MuiPopover-paper': {
          p: 1,
          mt: 3,
          borderRadius: 4,
          border: `2px solid ${slot.color}`,
          bgcolor: bgColor,
          overflow: 'visible',
          boxShadow: 'none',
          '&::before': {
            content: '""',
            position: 'absolute',
            top: -10,
            left: '45%',
            width: '10%',
            aspectRatio: '1 / 1',
            backgroundColor: bgColor,
            transform: 'rotate(45deg)',
            borderRadius: 1,
            border: `2px solid ${slot.color}`,
            borderBottom: 'none',
            borderRight: 'none',
            zIndex: 1,
          },
        },
      }}
    >
      <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1 }}>
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faWrench} color={slot.color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: slot.color }}>
            {slot.type === 'preventive' ? 'Entretien préventif' : 'Maintenance corrective'}
          </Typography>
        </Box>
        
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faCar} color={slot.color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: slot.color }}>
            {`${slot.vehicle.marque} ${slot.vehicle.modele} (${slot.vehicle.immatriculation})`}
          </Typography>
        </Box>

        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faCalendar} color={slot.color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: slot.color }}>
            {format(new Date(slot.start), 'dd/MM/yyyy', { locale: fr })}
          </Typography>
        </Box>

        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faClock} color={slot.color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: slot.color }}>
            {`${slot.startHeure}:${slot.startMinute} - ${slot.endHeure}:${slot.endMinute}`}
          </Typography>
        </Box>

        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faHourglassHalf} color={slot.color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: slot.color }}>
            {`Durée: ${slot.estimatedDuration ? `${Math.floor(slot.estimatedDuration)}h${slot.estimatedDuration % 1 ? '30' : '00'}` : '--'}`}
          </Typography>
        </Box>

        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faUser} color={slot.color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: slot.color }}>
            {slot.technician}
          </Typography>
        </Box>

        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <FontAwesome icon={faCircleCheck} color={slot.color} width={13} />
          <Typography sx={{ fontSize: pxToRem(12), color: slot.color }}>
            {getStatusLabel(slot.status)}
          </Typography>
        </Box>

        <ConditionalComponent 
          isValid={Boolean(slot.interventionDetails)}
        >
          <Box sx={{ 
            mt: 1,
            pt: 1,
            borderTop: `1px solid ${slot.color}`,
            display: 'flex',
            flexDirection: 'column',
            gap: 0.5
          }}>
            <Typography 
              sx={{ 
                fontSize: pxToRem(11), 
                color: slot.color,
                fontWeight: 600
              }}
            >
              Détails de l&apos;intervention
            </Typography>
            <Typography sx={{ fontSize: pxToRem(11), color: slot.color }}>
              {slot.interventionDetails}
            </Typography>
          </Box>
        </ConditionalComponent>
      </Box>
    </Popover>
  );
} 