import React from 'react';
import { Box, Typography, Card, LinearProgress, Grid, alpha } from '@mui/material';
import { styled } from '@mui/material/styles';

// Define types
interface LeaveBalance {
  type: string;
  used: number;
  total: number;
  color: string;
}

interface LeaveBalanceCardProps {
  balances: LeaveBalance[];
}

// Styled components
const BalanceCard = styled(Card)(({ theme }) => ({
  padding: theme.spacing(2),
  borderRadius: 12,
  border: `1px solid ${theme.palette.divider}`,
  boxShadow: 'none',
  backgroundColor: theme.palette.background.paper,
  height: '100%',
  display: 'flex',
  flexDirection: 'column',
  justifyContent: 'space-between',
  transition: 'all 0.3s ease',
  '&:hover': {
    boxShadow: '0 4px 20px 0 rgba(0,0,0,0.05)',
    backgroundColor: alpha(theme.palette.primary.main, 0.02),
    borderColor: alpha(theme.palette.primary.main, 0.2),
  }
}));

const ProgressLabel = styled(Box)({
  display: 'flex',
  justifyContent: 'space-between',
  marginBottom: 4,
});

export default function LeaveBalanceCard({ balances }: LeaveBalanceCardProps) {
  return (
    <Grid container sx={{ height: '100%' }}>
      <Grid item xs={12}>
        <BalanceCard>
          <Typography variant="h6" fontWeight={600} sx={{ mb: 2, color: (theme) => theme.palette.primary.main }}>
            Solde de congés restants
          </Typography>
          {balances.map((balance) => {
            const remaining = balance.total;
            
            // Determine color based on remaining days
            let statusColor = '#FF3434'; // Red for 0 days
            if (remaining > 10) {
              statusColor = '#36B37E'; // Green for more than 10 days
            } else if (remaining > 5) {
              statusColor = '#F9B90B'; // Yellow for 5-10 days
            } else if (remaining > 0) {
              statusColor = '#FF9800'; // Orange for 1-5 days
            }

            return (
              <Box 
                key={balance.type} 
                sx={{ 
                  mb: 2,
                  p: 1,
                  borderRadius: 1,
                  cursor: 'pointer',
                  transition: 'all 0.2s',
                  backgroundColor: alpha(statusColor, 0.1),
                  border: `1px solid ${alpha(statusColor, 0.2)}`,
                  '&:hover': {
                    backgroundColor: alpha(statusColor, 0.15),
                    transform: 'translateX(4px)',
                  },
                }}
              >
                <ProgressLabel>
                  <Typography 
                    variant="body2" 
                    fontWeight={500} 
                    sx={{ 
                      color: 'text.primary',
                      fontSize: '0.9rem'
                    }}
                  >
                    {balance.type}
                  </Typography>
                  <Typography 
                    variant="body2" 
                    fontWeight={700} 
                    sx={{ 
                      color: statusColor,
                      fontSize: '0.9rem'
                    }}
                  >
                    {remaining} jours
                  </Typography>
                </ProgressLabel>

                <Box sx={{ position: 'relative', mt: 1 }}>
                  <LinearProgress
                    variant="determinate"
                    value={100}
                    sx={{
                      height: 6,
                      borderRadius: 3,
                      backgroundColor: alpha(statusColor, 0.2),
                      '& .MuiLinearProgress-bar': {
                        borderRadius: 3,
                        backgroundColor: statusColor,
                      }
                    }}
                  />
                </Box>
              </Box>
            );
          })}
        </BalanceCard>
      </Grid>
    </Grid>
  );
} 