import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import { alpha, useTheme } from '@mui/material/styles';
import SvgColor from '@/shared/components/svg-color';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';

import TableActionColumn from '@/shared/components/table/table-action-column';

import { getStatusColor, StatusChip } from '../../../rh/styles';
import { IDemandeAbsence } from '@/shared/types/leave';
import StatusSelect from '@/shared/components/dialog/statusSelect';
import { AbsenceStyledRow } from '../styles';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

const LEAVE_STATUS = [
  {
    value: 'En attente',
    label: 'En attente',
    color: (theme: any) => theme.palette.primary.main,
    backgroundColor: (theme: any) => alpha(theme.palette.primary.main, 0.1)
  },
  {
    value: 'Approuvée',
    label: 'Approuvée',
    color: '#36B37E',
    backgroundColor: '#E9FBF0'
  },
  {
    value: 'Refusée',
    label: 'Refusée',
    color: '#FF3434',
    backgroundColor: '#FFE1E1'
  },
];

const TOOLTIP_TITLES = {
  edit: "Modifier demande",
  active: "Demande active",
  inactive: "Demande inactive",
};

type Props = {
  handleEdit: (absence: IDemandeAbsence) => void;
  handleDetails: (absence: IDemandeAbsence) => void;
  selected: boolean;
  row: IDemandeAbsence;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
};

export default function AbsenceTableRow({
  row,
  handleEdit,
  handleDetails,
  selected,
  onSelectRow,
  columns,
}: Props) {
  const theme = useTheme();

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'dateSubmission':
      case 'dateDebut':
      case 'dateFin':
      case 'dateTraitement':
        return value ? formatDate(value, dateFormat.isoDate) : '-';

      case 'status':
        // Display status as a non-interactive chip
        return (
          <StatusChip
            status={value}
            label={value}
            theme={theme}
          />
        );

      case 'modeReception':
        return value || '-';

      default:
        return value || '-';
    }
  };

  return (
    <>
      <AbsenceStyledRow
        hover
        style={{ cursor: 'pointer' }}
      >
        {columns
          .filter((column) => column.id !== 'isActive')
          .map((column, index) => (
            <TableCell
              onDoubleClick={() => handleDetails(row)}
              key={column.id}
              align={column.align || (column.id === 'daysCount' ? 'center' : 'left')}
              sx={{
                whiteSpace: 'nowrap',
                py: column.id === 'status' ? 0 : 'inherit',
                ...(index === 0 && {
                  position: 'relative',
                })
              }}
            >
              <ConditionalComponent isValid={index === 0}>
                <SvgColor
                  src="/assets/icons/ic_border.svg"
                  sx={{
                    position: 'absolute',
                    top: 0,
                    left: -4,
                    height: '100%',
                    color: getStatusColor(row.status),
                  }}
                />
              </ConditionalComponent>
              {renderCellContent(column.id, row[column.id as keyof IDemandeAbsence])}
            </TableCell>
          ))}

        <TableCell
          align="right"
          sx={{ px: 1, whiteSpace: 'nowrap' }}
          onClick={(e) => e.stopPropagation()}
        >
          <TableActionColumn
            handleEdit={() => handleEdit(row)}
            isActive={row.isActive}
            tooltipTitles={TOOLTIP_TITLES}
          />
        </TableCell>
      </AbsenceStyledRow>
    </>
  );
}