'use client';

import { useEffect, useState } from 'react';
import { _documentsRows } from '@/shared/_mock/_usagers';
import FontAwesome from '@/shared/components/fontawesome';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { TrajetStyledRow } from '@/shared/sections/contract/styles';
import { ActionsIconButton, iconButtonStyles, StyledTableContainer } from '@/shared/theme/css';
import { faPenToSquare, faTrashCan, faCheck, faTimes } from '@fortawesome/free-solid-svg-icons';
import { TableBody, TableCell, IconButton, Table, TextField } from '@mui/material';
import ConfirmDialog from '@/shared/components/dialog/confirmDialog';
import { StyledTextField } from '@/shared/components/table/styles';
import SvgColor from '@/shared/components/svg-color';
import { Stack } from '@mui/system';
import { IPassengerList } from '@/shared/types/passenger';
import TableNoData from '@/shared/components/table/table-no-data';

interface DepotDocumentFormProps {
  usager?: IPassengerList
}

export default function DepotDocumentForm({ usager }: DepotDocumentFormProps) {
  const [documents, setDocuments] = useState(_documentsRows);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [documentToDelete, setDocumentToDelete] = useState<string | null>(null);
  const [editingId, setEditingId] = useState<string | null>(null);
  const [editingLabel, setEditingLabel] = useState('');

  useEffect(() => {
    if (usager?.email) {
      setDocuments(_documentsRows);
    } else {
      setDocuments([]);
    }
  }, [usager]);

  const handleDeleteDocument = (id: string) => {
    setDocumentToDelete(id);
    setDeleteDialogOpen(true);
  };

  const confirmDelete = () => {
    if (documentToDelete !== null) {
      setDocuments(documents.filter(doc => doc.id !== documentToDelete));
    }
    setDeleteDialogOpen(false);
    setDocumentToDelete(null);
  };

  const cancelDelete = () => {
    setDeleteDialogOpen(false);
    setDocumentToDelete(null);
  };

  const handleEditDocument = (id: string, currentLabel: string) => {
    setEditingId(id);
    setEditingLabel(currentLabel);
  };

  const handleSaveEdit = (id: string) => {
    setDocuments(documents.map(doc => 
      doc.id === id ? { ...doc, label: editingLabel } : doc
    ));
    setEditingId(null);
    setEditingLabel('');
  };
  const handleCancelEdit = () => {
    setEditingId(null);
    setEditingLabel('');
  };

  return (<>
          <Stack flexGrow={0} direction="row" justifyContent="flex-end">
          <CustomTooltip title="Importer Document" arrow>
            <IconButton size="small" sx={iconButtonStyles}>
            <SvgColor src="/assets/icons/form/ic_upload.svg" sx={{ width: '16px',height: '16px' }} />
            </IconButton>
          </CustomTooltip>
        </Stack>
    <StyledTableContainer>
      <Table stickyHeader sx={{ mt: 1, minWidth: 960 }}>
        <TableBody>
        {documents.length === 0 ? (
            <TableNoData
              notFound={true} 
              colSpan={6}
            />
          ) : (
          documents.map((row, index) => (
            <TrajetStyledRow key={index}>
              <TableCell align="center">{row.id}</TableCell>
              <TableCell align="center">
                {editingId === row.id ? (
                  <StyledTextField
                    sx={{width:'80%'}}
                    value={editingLabel} 
                    onChange={(e) => setEditingLabel(e.target.value)}
                  />
                ) : (
                  row.label
                )}
              </TableCell>
              <TableCell align="center">{row.usager}</TableCell>
              <TableCell align="center">{row.dateImportation}</TableCell>
              <TableCell align="center">{row.dateFin}</TableCell>
              <TableCell align="center">
                {editingId === row.id ? (
                  <>
                    <CustomTooltip title="Save" arrow>
                      <IconButton
                        onClick={() => handleSaveEdit(row.id)}
                        size="small"
                        sx={ActionsIconButton}
                      >
                        <FontAwesome icon={faCheck} width={16} />
                      </IconButton>
                    </CustomTooltip>
                    <CustomTooltip title="Cancel" arrow>
                      <IconButton
                        onClick={handleCancelEdit}
                        size="small"
                        sx={ActionsIconButton}
                      >
                        <FontAwesome icon={faTimes} width={16} />
                      </IconButton>
                    </CustomTooltip>
                  </>
                ) : (
                  <CustomTooltip title="Edit Document" arrow>
                    <IconButton
                      onClick={() => handleEditDocument(row.id, row.label)}
                      size="small"
                      sx={ActionsIconButton}
                    >
                      <FontAwesome icon={faPenToSquare} width={14} />
                    </IconButton>
                  </CustomTooltip>
                )}
                <CustomTooltip title="Delete Document" arrow>
                  <IconButton
                    onClick={() => handleDeleteDocument(row.id)}
                    size="small"
                    sx={ActionsIconButton}
                  >
                    <FontAwesome icon={faTrashCan} width={14} />
                  </IconButton>
                </CustomTooltip>
              </TableCell>
            </TrajetStyledRow>
          ))
        )}
        </TableBody>
      </Table>
      <ConfirmDialog
        open={deleteDialogOpen}
        title="Confirm Delete"
        content="Are you sure you want to delete this document?"
        onConfirm={confirmDelete}
        onCancel={cancelDelete}
      />
    </StyledTableContainer>
    </>
  );
}

