import React from "react"
import { Table, TableHead, TableBody, TableCell, Typography } from "@mui/material"
import FontAwesome from "@/shared/components/fontawesome"
import { faCheck, faClock } from "@fortawesome/free-solid-svg-icons"
import { Box } from "@mui/system"
import {
  HeaderCell,
  HeaderCellWidthColor,
  HeaderRow,
  SensARCheckbox,
  StyledEstablishmentRow,
  StyledFirstRow,
  StyledSecondRow,
  StyledSubTableRow,
  StyledTableRow,
} from "./style"
import { StyledTableContainer } from "@/shared/theme/css"
import { pxToRem } from "@/shared/theme/typography"

interface ScheduleTableProps {
  title: string
  defaultTime?: string
  mode: "time" | "checkbox"
  onSelectionChange?: (selection: {
    week: "pair" | "impair"
    day: string
    type: "all" | "ret"
    checked: boolean
  }) => void
  selectedDays?: {
    pair: { [key: string]: { all: boolean; ret: boolean } }
    impair: { [key: string]: { all: boolean; ret: boolean } }
  }
  isInEstablishment?: boolean
  isRepresentative?: boolean
}

const WEEKDAYS = ["monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday"]

export default function ScheduleTable({
  title,
  defaultTime = "08:00 AM",
  mode = "time",
  onSelectionChange,
  selectedDays = {
    pair: {},
    impair: {},
  },
  isInEstablishment,
  isRepresentative,
}: ScheduleTableProps) {
  const renderCell = (week: "pair" | "impair", day: string, type: "all" | "ret") => {
    if (mode === "time") {
      return (
        <TableCell>
          <Box sx={{ whiteSpace: "nowrap" }} display="flex" alignItems="center">
            <FontAwesome icon={faClock} color="primary.main" width={12} sx={{ mr: 1 }} />
            <Typography variant="caption">{defaultTime}</Typography>
          </Box>
        </TableCell>
      )
    }

    const isChecked = selectedDays[week]?.[day]?.[type] || false
    const cellSize =  isRepresentative ? "16px" : "20px";
    return (
      <TableCell sx={{ p: isRepresentative ? "4px" : "8px" }} align="center">
        <Box sx={{ position: "relative" }}>
          <SensARCheckbox
            checked={isChecked}
            sx={{width:cellSize,height:cellSize }}
            onChange={(e) => {
              onSelectionChange?.({
                week,
                day,
                type,
                checked: e.target.checked,
              })
            }}
            size="small"
          />

          <FontAwesome
            sx={{
              color: "white",
              position: "absolute",
              top: isRepresentative ? "58%" : "50%",
              left: "50%",
              transform: "translate(-50%, -50%)",
            }}
            width={isRepresentative ? 12 : 16}
            icon={faCheck}
          />
        </Box>
      </TableCell>
    )
  }

  const CellComponent = isInEstablishment ? HeaderCell : HeaderCellWidthColor
  const FirstRow = isInEstablishment ? StyledEstablishmentRow : StyledFirstRow
  const periods = isInEstablishment ? ["Matin","Soir"] : ["All", "Ret"]
  const fontSize =  isRepresentative ? pxToRem(13) : undefined;

  return (
    <>
      <Typography
        variant={isRepresentative ? "h6" : "h5"}
        sx={{
          mt: isInEstablishment ? 0 : 3,
          mb: isInEstablishment ? 1 : 3,
          color: "primary.main",
          fontWeight: (theme) => theme.typography.fontWeightBold,
        }}
      >
        {title}
      </Typography>
      <StyledTableContainer>
        <Table stickyHeader sx={{ mt: 1 }}>
          <TableHead>
            <StyledTableRow >
              <CellComponent sx={{ textAlign: "left", px: 2,fontSize:fontSize }} rowSpan={2}>
                {isInEstablishment ? "Jours" : "Semaines"}
              </CellComponent>
              {WEEKDAYS.map((day) => (
                <CellComponent sx={{ fontSize:fontSize }} key={day} colSpan={2}>
                  {day}
                </CellComponent>
              ))}
            </StyledTableRow>
            <StyledSubTableRow>
              {WEEKDAYS.map((day) => (
                <React.Fragment key={`${day}-subheaders`}>
                  {periods.map((period) => (
                    <CellComponent sx={{ fontSize:fontSize,px:'4px' }} key={`${day}-${period}`}>{period}</CellComponent>
                  ))}
                </React.Fragment>
              ))}
            </StyledSubTableRow>
          </TableHead>
          <TableBody>
            <FirstRow>
              <HeaderRow sx={{ fontSize:fontSize,pl:1,pr:0 }}>{isInEstablishment ? "Horaires" : "Semaine Paire"}</HeaderRow>
              {WEEKDAYS.map((day) => (
                <React.Fragment key={`pair-${day}`}>
                  {renderCell("pair", day, "all")}
                  {renderCell("pair", day, "ret")}
                </React.Fragment>
              ))}
            </FirstRow>
            {!isInEstablishment ? (
              <StyledSecondRow>
                <HeaderRow sx={{ fontSize:fontSize,pl:1,pr:0 }}>Semaine Impaire</HeaderRow>
                {WEEKDAYS.map((day) => (
                  <React.Fragment key={`impair-${day}`}>
                    {renderCell("impair", day, "all")}
                    {renderCell("impair", day, "ret")}
                  </React.Fragment>
                ))}
              </StyledSecondRow>
            ):<></>}
          </TableBody>
        </Table>
      </StyledTableContainer>
    </>
  )
}

