"use client"

import React, { useCallback, useMemo } from "react"
import { Polyline } from "@react-google-maps/api"
import { Popover, Stack, IconButton } from "@mui/material"
import FontAwesome from "../fontawesome"
import CustomTooltip from "../tooltips/tooltip-custom"
import { faCircleCheck, faEyeSlash, faPenToSquare, faTrashCan } from "@fortawesome/free-solid-svg-icons"
import { iconButtonStyles } from "@/shared/theme/css"
import { LatLng } from "@/shared/types/Maps"

interface MapPolylineProps {
  path: LatLng[]
  color: string
  index: number
  onPolylineClick: (event: google.maps.MapMouseEvent, index: number) => void
}

export const MapPolyline: React.FC<MapPolylineProps> = ({ path, color, index, onPolylineClick }) => {
  const [anchorPosition, setAnchorPosition] = React.useState<{ top: number; left: number } | null>(null)
  const [isPopoverOpen, setIsPopoverOpen] = React.useState(false)

  const handlePolylineClick = useCallback(
    (e: google.maps.MapMouseEvent) => {
      const mouseEvent = e.domEvent as MouseEvent      
      setAnchorPosition({
        top: mouseEvent.clientY,
        left: mouseEvent.clientX
      })
      setIsPopoverOpen(true)
      onPolylineClick(e, index)
    },
    [index, onPolylineClick]
  )
  const handleClosePopover = () => {
    setIsPopoverOpen(false)
    setAnchorPosition(null)
  }

  const handleEditRoute = () => {
    handleClosePopover()
  }

  const handleDeleteRoute = () => {
    handleClosePopover()
  }

  const handleRouteInfo = () => {
    handleClosePopover()
  }
  const lineSymbol = useMemo(
    () => ({
      path: "M 0,-1 0,1",
      strokeOpacity: 1,
      scale: 4,
    }),
    [],
  )

  const polylineOptions: google.maps.PolylineOptions = {
    strokeColor: color,
    strokeWeight: 3,
    clickable: true,
    ...(index === 1 && {
      strokeOpacity: 0,
      icons: [{
        icon: lineSymbol,
        offset: "0",
        repeat: "20px",
      }]
    })
  }

  return (
    <>
      <Polyline
        path={path}
        options={polylineOptions}
        onClick={handlePolylineClick}
      />
      
      <Popover
        open={isPopoverOpen}
        anchorReference="anchorPosition"
        anchorPosition={anchorPosition || undefined}
        onClose={handleClosePopover}
        transformOrigin={{
          vertical: 'top',
          horizontal: 'left',
        }}
        slotProps={{
          paper: {
            sx: {
              borderRadius: 3,
              p: 1,
              pointerEvents: 'auto',
              boxShadow: '0 4px 20px rgba(0,0,0,0.2)'
            }
          }
        }}
      >
        <Stack direction="row" spacing={1}>
          <CustomTooltip title={`Modifier Circuit`} arrow>
            <IconButton onClick={handleEditRoute} size="small" sx={iconButtonStyles}>
              <FontAwesome icon={faPenToSquare} width={14} />
            </IconButton>
          </CustomTooltip>

          <CustomTooltip title={`Valider Circuit`} arrow>
            <IconButton onClick={handleEditRoute} size="small" sx={iconButtonStyles}>
              <FontAwesome icon={faCircleCheck} width={14} />
            </IconButton>
          </CustomTooltip>

          <CustomTooltip title={`cacher Circuit`} arrow>
            <IconButton onClick={handleRouteInfo} size="small" sx={iconButtonStyles}>
              <FontAwesome icon={faEyeSlash} width={14} />
            </IconButton>
          </CustomTooltip>

          <CustomTooltip title={`Supprimer Circuit`} arrow>
            <IconButton onClick={handleDeleteRoute} size="small" sx={iconButtonStyles}>
              <FontAwesome icon={faTrashCan} width={14} />
            </IconButton>
          </CustomTooltip>
        </Stack>
      </Popover>
    </>
  )
}
