import { useEffect, useState } from 'react';
import { faUserTie, faStickyNote } from '@fortawesome/free-solid-svg-icons';
import type { FormFieldType } from '@/shared/types/common';
import type { IAbsence, IPassenger } from '@/shared/types/absence';
import { _SENS_AR } from '@/shared/_mock';
import { usePassengerStore } from '@/shared/api/stores/passengerStore';
import { FormikProps } from 'formik';

export type OptionType = {
  value: string;
  label: string;
};

const RECEPTION_MODES = [
  { value: 'SMS', label: 'SMS' },
  { value: 'EMAIL', label: 'Email' },
  { value: 'APPLICATION', label: 'Application mobile' }
];

export const _OPTION_DEMANDE = [
  { value: 'Traité', label: 'Traité' },
  { value: 'Non traité', label: 'Non traité' },
  { value: 'Annulé', label: 'Annulé' },
  { value: 'En cours', label: 'En cours' }
];

export const useFormFieldsAbsence = (optionDemande: OptionType[]) => {
  const { searchPassengers, passengers } = usePassengerStore();

  useEffect(() => {
    searchPassengers();
  }, [searchPassengers]);

  const fields: FormFieldType<IAbsence>[] = [
    {
      name: 'usager' as keyof IAbsence,
      label: 'Usager',
      type: 'select',
      required: true,
      options: passengers.map(passenger => ({
        value: passenger.id.toString(),
        label: `${passenger.firstName} ${passenger.lastName}`
      })),
      getOptionLabel: (option: IPassenger) => {
        if (!option) return '';
        if (typeof option === 'string') return option;
        return `${option.firstName} ${option.lastName}`;
      },
      renderValue: (selected:number) => {
        if (!selected) return '';
        const passenger = passengers.find(p => p.id.toString() === selected.toString());
        return passenger ? `${passenger.firstName} ${passenger.lastName}` : '';
      },
      onChange: (selected:number, form:FormikProps<IAbsence>) => {
        if (!selected) return;
        
        const selectedPassenger = passengers.find(p => p.id.toString() === selected.toString());
        if (selectedPassenger) {
          form.setFieldValue('usager', {
            id: selectedPassenger.id,
            firstName: selectedPassenger.firstName,
            lastName: selectedPassenger.lastName,
            establishmentName: selectedPassenger.establishmentName,
            departmentName: selectedPassenger.departmentName
          });
          form.setFieldValue('etablissement', selectedPassenger.establishmentName);
          form.setFieldValue('departement', selectedPassenger.departmentName);
        }
      }
    },
    {
      name: 'dateDebut',
      label: 'Date de début',
      type: 'date',
      required: true,
    },
    {
      name: 'dateFin',
      label: 'Date de fin',
      type: 'date',
      required: true,
    },
    {
      name: 'par',
      label: 'Par',
      placeholder: 'Entrez un nom',
      required: true,
      type: 'text',
      icon: faUserTie
    },
    {
      name: 'sensAR',
      label: 'Trajets concernés',
      type: 'checkbox-group',
      options: [
        { value: 'aller', label: 'Aller' },
        { value: 'retour', label: 'Retour' }
      ],
      required: true,
    },
    {
      name: 'modeReception',
      label: 'Mode de réception',
      type: 'select',
      options: RECEPTION_MODES,
      required: true,
    },
    {
      name: 'previentLe',
      label: 'Quand',
      type: 'date',
      required: true,
    },
    {
      name: 'etat',
      label: 'Etat',
      type: 'select',
      options: optionDemande,
      disabled: true,
      required: true
    },
    {
      name: 'observation',
      label: "Observation",
      placeholder: "Ajoutez une note ou une observation",
      type: 'textarea',
      icon: faStickyNote,
    },
  ];

  return fields;
};
