import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import { IAgent } from '@/shared/types/agent';
import TableActionColumn from '@/shared/components/table/table-action-column';
import SvgColor from '@/shared/components/svg-color';
import { TableRow } from '@mui/material';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { AgentStyledRow, StatusChip, getStatusColor } from '../../styles';
import { _AGENT_STATUS } from '@/shared/_mock/_agent';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';




type Props = {
  handleEditAgent: (agent: IAgent) => void;
  handleDetailsAgent: (agent: IAgent) => void;
  handleCopyAgent: (agent: IAgent) => void;
  selected: boolean;
  row: IAgent;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: IAgent, newStatus: string, field: keyof IAgent) => void;
};

const TOOLTIP_TITLES = {
  edit: 'Modifier agent',
  duplicate: 'Copier agent',
  active: 'Agent actif',
  inactive: 'Agent inactif',
};

export default function AgentTableRow({
  row,
  handleEditAgent,
  handleDetailsAgent,
  handleCopyAgent,
  selected,
  onSelectRow,
  columns,
  onStatusChange
}: Props) {

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'birthDate':
        return formatDate(value, dateFormat.isoDate);
      case 'address':
        return value.address;
      case 'status':
        return (
          <StatusMenu
          value={value}
          options={_AGENT_STATUS}
          onChange={(newStatus) => onStatusChange(row, newStatus, 'status')}
          chipComponent={StatusChip}
          sx={{ width: "120px" }}
        />
      
        );
      case 'permissions':
        return Array.isArray(value) ? value.join(', ') : value;
      default:
        return value;
    }
  };

  return (
    <AgentStyledRow hover style={{ cursor: 'pointer' }}>

      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any,index:number) => (
          <TableCell
            onDoubleClick={() => handleDetailsAgent(row)}
            key={column.id}
            align={column.align}
            sx={{ 
              whiteSpace: 'nowrap', 
              py: column.id === 'status' ? 0 : 'inherit',
              maxWidth: '200px',
              overflow: 'hidden',
              textOverflow: 'ellipsis',
              ...(index === 0 && {
                position: 'relative',
              })
            }}
          >
            <ConditionalComponent isValid={index === 0}>
              <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{
                  position: 'absolute',
                  top: 0,
                  left: -4,
                  height: '100%',
                  color: getStatusColor(row.status),
                }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IAgent])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditAgent(row)}
          onDuplicate={() => handleCopyAgent(row)}
          onToggle={(checked) => void 0}
          isActive={row.isActive}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </AgentStyledRow>
  );
}