import React from 'react';
import { TeamList } from '@/shared/types/equipe';
import EquipeTableRow from '../view/equipe-table-row';
import TableManager from '@/shared/components/table/table-manager';
import { TableColumn } from '@/shared/types/common';

interface EquipeTableProps {
  filteredData: TeamList[];
  table: ReturnType<typeof import('@/shared/components/table/use-tablets').default>;
  tableHead: TableColumn[];
  notFound: boolean;
  filters: Record<string, { value: string; condition: string }>;
  onFilterChange: (key: string, value: string, condition: string) => void;
  handleEdit: (equipe: TeamList) => void;
  handleCopy: (equipe: TeamList) => void;
  handleView: (equipe: TeamList) => void;
  onStatusChange: (row: TeamList, newStatus: string, field: keyof TeamList) => void;
  totalTeams: number;
  onPageChange: (event: React.MouseEvent<HTMLButtonElement> | null, page: number) => void;
  page: number;
  onRowsPerPageChange: (event: React.ChangeEvent<HTMLInputElement>) => void;
  rowsPerPage: number;
  sortBy: string;
  sortDirection: 'asc' | 'desc';
  loading?: boolean;
  refreshData: () => void;
}

export default function EquipeTable({
  filteredData,
  table,
  tableHead,
  notFound,
  filters,
  onFilterChange,
  handleEdit,
  handleCopy,
  handleView,
  onStatusChange,
  totalTeams,
  onPageChange,
  page,
  onRowsPerPageChange,
  rowsPerPage,
  sortBy,
  sortDirection,
  loading,
  refreshData
}: EquipeTableProps) {
  const renderRow = (row: TeamList) => (
    <EquipeTableRow
      key={row.id}
      row={row}
      handleEditEquipe={() => handleEdit(row)}
      handleCopyEquipe={() => handleCopy(row)}
      handleDetailsEquipe={() => handleView(row)}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead.slice(0, -1)}
      onStatusChange={onStatusChange}
      refreshData={refreshData}
    />
  );

  return (
    <TableManager
      filteredData={filteredData}
      table={table}
      tableHead={tableHead}
      notFound={notFound}
      filters={filters}
      onFilterChange={onFilterChange}
      renderRow={renderRow}
      onPageChange={onPageChange}
      page={page}
      onRowsPerPageChange={onRowsPerPageChange}
      rowsPerPage={rowsPerPage}
      count={totalTeams}
      sortBy={sortBy}
      sortDirection={sortDirection}
      loading={loading}
    />
  );
}