"use client"

import Checkbox from "@mui/material/Checkbox"
import TableCell from "@mui/material/TableCell"
import SvgColor from "@/shared/components/svg-color"
import type { TableColumn } from "@/shared/types/common"
import { type IEmailTemplate, EmailTemplateCategory } from "@/shared/types/email-template"
import { ContratStyledRow } from "@/shared/sections/contract/styles"
import { formatDate } from "@/utils/format-time"
import { Chip, IconButton, Tooltip, Box } from "@mui/material"
import { Edit as EditIcon, Archive as ArchiveIcon, FileDownload as ExportIcon } from "@mui/icons-material"
import TableActionColumn from "@/shared/components/table/table-action-column"
import CustomTooltip from "@/shared/components/tooltips/tooltip-custom"
import { ActionsIconButton } from "@/shared/theme/css"
import { faFileExport } from "@fortawesome/free-solid-svg-icons"
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"

type Props = {
  handleEdit: (template: IEmailTemplate) => void
  handleDetails: (template: IEmailTemplate) => void
  handleArchive: (template: IEmailTemplate) => void
  handleExport: (template: IEmailTemplate) => void
  selected: boolean
  row: IEmailTemplate
  dense: boolean
  onSelectRow: VoidFunction
  columns: TableColumn[]
}

const TOOLTIP_TITLES = {
  edit: 'Modifier modèle',
  active: "Actif",
  inactive: "Inactif",
  export: "Exporter",
}

const getCategoryColor = (category: EmailTemplateCategory) => {
  switch (category) {
    case EmailTemplateCategory.NOTIFICATION_RH:
      return {
        color: "primary",
        backgroundColor: "primary.lighter",
        borderColor: "primary.light",
      }
    case EmailTemplateCategory.COMMUNICATION_INTERNE:
      return {
        color: "info",
        backgroundColor: "info.lighter",
        borderColor: "info.light",
      }
    case EmailTemplateCategory.RAPPEL:
      return {
        color: "warning",
        backgroundColor: "warning.lighter",
        borderColor: "warning.light",
      }
    case EmailTemplateCategory.INFORMATION:
      return {
        color: "success",
        backgroundColor: "success.lighter",
        borderColor: "success.light",
      }
    default:
      return {
        color: "default",
        backgroundColor: "grey.200",
        borderColor: "grey.300",
      }
  }
}

export default function EmailTemplateTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleArchive,
  handleExport,
  selected,
  onSelectRow,
  columns,
}: Props) {
  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "dateCreation":
        return formatDate(value)
      case "categorie":
        const categoryColors = getCategoryColor(value as EmailTemplateCategory)
        return (
          <Chip
            label={value.replace(/_/g, " ")}
            size="small"
              variant="outlined"
              sx={{
              '&.MuiChip-root': {
              borderRadius: '8px',
              padding: '9px 0',
              height: '100%',
              width: '200px',
              },
              color: `${categoryColors.color}.main`,
              backgroundColor: categoryColors.backgroundColor,
              borderColor: categoryColors.borderColor,
              fontWeight: "medium",
            }}
          />
        )
      default:
        return value
    }
  }

  return (
    <ContratStyledRow hover style={{ cursor: "pointer" }}>
      

      {columns
        .filter((column) => column.id !== "isArchive")
        .map((column: any,index:number) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: "nowrap", py: dense ? 0 : 2,
              ...(index === 0 && {
                position: 'relative',
              })
            }}
          >
            <ConditionalComponent isValid={index === 0}>
              <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{
                  position: 'absolute',
                  top: 0,
                  left: -4,
                  height: '100%',
                  color: "primary.main",
                }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IEmailTemplate])}
          </TableCell>
        ))}

      <TableCell
        sx={{ px: 1, whiteSpace: 'nowrap' }}
      >
        <Box sx={{ display: "flex", gap: 1 }}>
          <TableActionColumn
            handleEdit={() => handleEdit(row)}
            onToggle={() => handleArchive(row)}
            isActive={row.isArchive}
            tooltipTitles={TOOLTIP_TITLES}
          />
          <CustomTooltip title={"Exporter"} arrow>
            <IconButton onClick={() => handleExport(row)} size="small" sx={ActionsIconButton}>
              <FontAwesomeIcon icon={faFileExport} size="sm" />
            </IconButton>
          </CustomTooltip>
        </Box>
      </TableCell>
    </ContratStyledRow>
  )
}
