import { useState, useEffect } from 'react';
import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { TableColumn } from '@/shared/types/common';
import React from 'react';
import type { AgencyList, AgencyPageFilterParams, SectorPageFilterParams } from '@/shared/types/infrastructure';
import { useInfrastructureStore } from '@/shared/api/stores/admin-service/infrastructureStore';

export const DEFAULT_TABLE_HEAD: TableColumn[] = [
    { id: "name", label: "Agence", type: "text" },
    { id: "address", label: "Adresse", type: "text" },
    { id: "description", label: "Description", type: "text" },
    { id: "createdAt", label: "Date de création", type: "date" },
    { id: "", label: "Action" },
];

const agencyConfig: TableConfig<AgencyList> = {
    initialData: [],
    defaultTableHead: DEFAULT_TABLE_HEAD,
};

export function useAgencyTable() {
    const [params, setParams] = useState<AgencyPageFilterParams>({
        page: 0,
        size: 20,
        sortBy: 'name',
        sortDirection: 'desc'
    });

    const { agencyPage, agencyPageTotalElements, loading, error, fetchAgencyPage } = useInfrastructureStore();

    useEffect(() => {
        fetchAgencyPage(params);
    }, [params]);

    const tableManager = useTableManager<any>({
        ...agencyConfig,
        initialData: agencyPage || [],
    });

    useEffect(() => {
        if (tableManager.table.rowsPerPage > 0) {
            setParams(prev => ({
                ...prev,
                page: tableManager.table.page,
                size: tableManager.table.rowsPerPage,
            }));
        }
    }, [tableManager.table.page, tableManager.table.rowsPerPage]);

    const isValidDateInput = (value: any): boolean => {
        if (!value || typeof value !== 'string') return false;
        const date = new Date(value);
        return date instanceof Date && !isNaN(date.getTime());
    };
    
    const isValidFilterValue = (value: any): boolean => {
        return value !== null && value !== undefined && value !== '';
    };

    useEffect(() => {
        const baseParams: AgencyPageFilterParams = {
            page: tableManager.table.page || 0,
            size: tableManager.table.rowsPerPage || 20,
            sortBy: tableManager.table.orderBy || 'name',
            sortDirection: tableManager.table.order.toLowerCase() as 'asc' | 'desc'
        };

        const filterParams: Record<string, string | number> = {};

        Object.entries(tableManager.filters).forEach(([key, { value }]) => {
            if (key === 'createdAt' && isValidDateInput(value)) {
                const date = new Date(value);
                const year = date.getFullYear();
                const month = String(date.getMonth() + 1).padStart(2, '0');
                const day = String(date.getDate()).padStart(2, '0');
                filterParams[key] = `${year}-${month}-${day}T00:00:00`;
            } else if (isValidFilterValue(value)) {
                filterParams[key] = value;
            }
        });

        const newParams: AgencyPageFilterParams = {
            ...baseParams,
            ...filterParams
        };

        const paramsChanged = JSON.stringify(newParams) !== JSON.stringify(params);
        if (paramsChanged) {
            setParams(newParams);
        }
    }, [tableManager.filters, tableManager.table.orderBy, tableManager.table.order]);

    const handlePageChange = (event: React.MouseEvent<HTMLButtonElement> | null, newPage: number) => {
        setParams(prev => ({
            ...prev,
            page: newPage
        }));
    };

    const handleRowsPerPageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const size = parseInt(event.target.value, 10);
        if (!isNaN(size) && size > 0) {
            setParams(prev => ({
                ...prev,
                size,
                page: 0
            }));
        }
    };

    return {
        ...tableManager,
        loading,
        error,
        totalAgencies: agencyPageTotalElements || 0,
        onPageChange: handlePageChange,
        page: params.page,
        onRowsPerPageChange: handleRowsPerPageChange,
        rowsPerPage: params.size,
        sortBy: params.sortBy || 'name',
        sortDirection: params.sortDirection || 'desc',
        filteredData: agencyPage
    };
}
