import React, { useState } from 'react';
import { Grid, Box, Typography } from '@mui/material';
import FontAwesome from '@/shared/components/fontawesome';
import { faClock } from '@fortawesome/free-solid-svg-icons';
import { RoadMapStyledTimePicker } from '@/shared/theme/css';
import { LocalizationProvider } from '@mui/x-date-pickers';
import { AdapterDateFns } from '@mui/x-date-pickers/AdapterDateFns';
import { pxToRem } from '@/shared/theme/typography';
import { AntSwitch } from '@/shared/components/table/styles';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { useTheme } from '@mui/material/styles';

interface TimePickerSectionProps {
  label: string;
}

const TimePickerSection: React.FC<TimePickerSectionProps> = ({ label }) => {
  const [active, setActive] = useState(true)
  const theme = useTheme();

  const tooltipTitles = {
    active: `Désactiver le trajet ${label.toLowerCase()}`,
    inactive: `Activer le trajet ${label.toLowerCase()}`,
  }

  const handleToggle = (checked: boolean) => {
    setActive(checked)
  }

return (
<LocalizationProvider dateAdapter={AdapterDateFns}>
    <Grid item xs={12} sm={3}>
    <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          mb: 2,
        }}
      >
        <Typography sx={{ fontWeight: "bold" }}>{label}</Typography>

        <CustomTooltip title={active ? tooltipTitles.active : tooltipTitles.inactive} arrow>
          <AntSwitch theme={theme} checked={active} onChange={(event) => handleToggle(event.target.checked)} size="small" />
        </CustomTooltip>
      </Box>

      <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
        <RoadMapStyledTimePicker
          label={`Départ ${label.toLowerCase()}`}
          slots={{
            openPickerIcon: () => <FontAwesome icon={faClock} sx={{ color: 'primary.main' }} />,
          }}
          views={['hours', 'minutes']}
          format="HH:mm"
          disabled={!active}
        />
        <RoadMapStyledTimePicker
          label={`Fin ${label.toLowerCase()}`}
          slots={{
            openPickerIcon: () => <FontAwesome icon={faClock} sx={{ color: 'primary.main' }} />,
          }}
          views={['hours', 'minutes']}
          format="HH:mm"
          disabled={!active}
        />
      </Box>
    </Grid>
  </LocalizationProvider>
)
}

const RoadMapTimePickerComponent: React.FC = () => (
  <>
    <Typography
      variant="h5"
      sx={{
        color: 'primary.main',
        fontSize: pxToRem(22),

        fontWeight: (theme) => theme.typography.fontWeightBold,
        m: 3,
      }}
    >
      Suivie des heures{' '}
    </Typography>
    <Box
      sx={{
        backgroundColor: 'white',
        borderRadius: 2,
        boxShadow: 1,
      }}
    >
      <Grid container spacing={4} sx={{ p: 3 }}>
        <TimePickerSection label="Matin" />

        <TimePickerSection label="Midi" />

        <TimePickerSection label="Après-midi" />

        <TimePickerSection label="Soir" />
      </Grid>
    </Box>
  </>
);

export default RoadMapTimePickerComponent;
