import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import { format } from 'date-fns';
import { useTheme } from '@mui/material/styles';
import SvgColor from '@/shared/components/svg-color';
import { ContratStyledRow, getStatusColor, StatusChip } from '../../contract/styles';
import { ICircuit } from '@/shared/types/circuit';
import { _CIRCUIT_STATUS, getEtatLabel } from '@/shared/_mock/_circuit';
import { AntSwitch } from '@/shared/components/table/styles';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { useBoolean } from '@/hooks';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import { getEtatColor } from '../utils/form-fields-circuit';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { _CONTRACT_STATUS } from '@/shared/_mock/_Contrat';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

type Props = {
  handleEditContract: (contrat: ICircuit) => void;
  handleDetailsContract: (contrat: ICircuit) => void;
  selected: boolean;
  row: ICircuit;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: ICircuit, newStatus: string, field: keyof ICircuit) => void;
};

export default function CircuitTableRow({
  dense,
  row,
  handleEditContract,
  handleDetailsContract,
  selected,
  onSelectRow,
  columns,
  onStatusChange
}: Props) {
  const theme = useTheme();
  const active = useBoolean(row.isActive);

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'dateDebut':
      case 'dateDernierAvenant':
      case 'dateFin':
        return formatDate(value,dateFormat.isoDate);
      case 'adresseEtablissement':
        return (
          row.adresseEtablissement.address
        );
      case 'etat':
        return (
          <StatusMenu
            value={value}
            options={_CIRCUIT_STATUS}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'etat')}
            chipComponent={StatusChip}
          />        
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>
      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetailsContract(row)}
            key={column.id}
            align={column.align || 'left'}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'etat' ? 0.2 : dense ? 1 : 2, ...(index === 0 && {
              position: 'relative',
            }), }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: 'absolute', top: 0, left: -4, height: '100%', color: getEtatColor(row.etat),
              }}
            />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof ICircuit])}
          </TableCell>
        ))}
      <TableCell
        align="center"
        sx={{
          display: 'flex',
          justifyContent: 'center',
          alignItems: 'center',
        }}
      >
        <CustomTooltip title={active.value ? 'Circuit actif' : 'Circuit inactif'} arrow>
          <AntSwitch
            theme={theme}
            checked={active.value}
            onChange={() => {
              active.onToggle();
            }}
            size="small"
          />
        </CustomTooltip>
      </TableCell>
    </ContratStyledRow>
  );
}
