import React from 'react';
import TableCell from '@mui/material/TableCell';
import { CONTRACT_STATUS, ContractList, ContractStatus } from '@/shared/types/contract';
import TableActionColumn from '@/shared/components/table/table-action-column';
import SvgColor from '@/shared/components/svg-color';
import { ContratStyledRow, getStatusColor, StatusChip } from '../styles';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { useContractStore } from '@/shared/api/stores/admin-service/contractStore';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

type Props = {
  handleEditContract: (contract: ContractList) => void;
  handleDetailsContract: (contract: ContractList) => void;
  handleCopyContract: (contract: ContractList) => void;
  selected: boolean;
  row: ContractList;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
};

const TOOLTIP_TITLES = {
  edit: 'Modifier contrat',
  duplicate: 'Copier contrat',
  active: 'Contrat actif',
  inactive: 'Contrat inactif',
};


export default function ContratTableRow({
  row,
  handleEditContract,
  handleDetailsContract,
  handleCopyContract,
  selected,
  onSelectRow,
  columns,
}: Props) {

  const { toogleActivation, updateContractStatus } = useContractStore();

  const handleToggleActivation = async (isActive: boolean) => {
    await toogleActivation(row.id, isActive);
  }

  const handleStatusChange = (newStatus: ContractStatus) => {
    updateContractStatus(row.id, newStatus);
  }

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'marketDate':
      case 'renewalDate':
      case 'endDate':
        return formatDate(value, dateFormat.isoDate);
      case 'status':
        return (
          <StatusMenu
            value={value}
            options={CONTRACT_STATUS}
            onChange={(newStatus) => handleStatusChange(newStatus as ContractStatus)}
            chipComponent={StatusChip}
            sx={{ width: "120px" }}
          />
        );
      case 'companyName':
        return row.agencyName;
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>

      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetailsContract(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'status' ? 0 : 'inherit', ...(index === 0 && {
              position: 'relative',
            }), }}
          >
            <ConditionalComponent
             isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: 'absolute', top: 0, left: -4, height: '100%', color: getStatusColor(row.status),
              }}
            />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof ContractList])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditContract(row)}
          onDuplicate={() => handleCopyContract(row)}
          onToggle={(checked) => handleToggleActivation(checked)}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}
