import type React from 'react';
import { Grid, Typography } from '@mui/material';
import { faFileArrowUp } from '@fortawesome/free-solid-svg-icons';
import { useFormikContext } from 'formik';
import FontAwesome from '@/shared/components/fontawesome';
import { centerFlexColumn } from '@/shared/theme/css';
import { TableColumn, TableType, type FormFieldType, type ModeType } from '@/shared/types/common';
import ActionButtons from '@/shared/components/form/buttons-action';
import type { IEtablissement, IResponsableEtablissement } from '@/shared/types/etablissement';
import {
  detailsEtabformFields,
  observationformFields,
  responsableEtabformFields,
} from '../utils/form-fields-establishment';
import { AddButton } from '@/shared/components/table/styles';
import FormContainer from '@/layouts/form/form-container';
import FormField from '@/shared/components/form/form-field';
import EditExportButtons from '@/shared/components/form/edit-export-buttons';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { EstablishmentRequest, EstablishmentResponsible } from '@/shared/types/establishment';

interface EtablissementTabProps {
  renderField: (field: FormFieldType<EstablishmentRequest>) => React.ReactNode;
  mode: ModeType;
  onClose: () => void;
  onEdit: () => void;
  etablissement: EstablishmentRequest;
  tableHead: TableColumn[];
  zoneOptions: {label: string, value: string}[];
  departmentOptions: {label: string, value: string}[];
}

export default function EstablishmentDetailsForm({
  mode,
  onClose,
  onEdit,
  renderField,
  etablissement,
  tableHead,
  zoneOptions,
  departmentOptions,
}: EtablissementTabProps) {
  const { values, setFieldValue, errors, touched, handleBlur, handleSubmit } = useFormikContext<EstablishmentRequest>();
  const isReadOnly = mode === 'view';

  const renderFieldResponsable = (field: FormFieldType<EstablishmentResponsible>) => (
    <FormField
      field={field}
      value={values.establishmentResponsible?.[field.name]}
      onChange={(name, value) => setFieldValue(`establishmentResponsible.${name}`, value)}
      error={
        touched.establishmentResponsible?.[field.name] && errors.establishmentResponsible?.[field.name]
          ? String(errors.establishmentResponsible[field.name])
          : undefined
      }
      isReadOnly={isReadOnly}
      onBlur={(e) => handleBlur({ target: { name: `establishmentResponsible.${field.name}` } })}
    />
  );

  return (
    <>
      <ConditionalComponent isValid={isReadOnly}>
        <EditExportButtons
          onEdit={onEdit}
          onExport={() => void 0}
          tooltipTitle={TableType.Etablissement}
          dataRow={etablissement}
          tableHead={tableHead}
        />
      </ConditionalComponent>
      <form onSubmit={handleSubmit}>
        <Grid container spacing={5}>
          {detailsEtabformFields(zoneOptions, departmentOptions).map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}
          <Grid
            item
            xs={12}
            sm={4}
            sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}
            key="importezDocument"
          >
            <Typography
              sx={{
                color: isReadOnly ? "text.disabled" : "primary.main",
                display: { sm: 'none', md: 'block' },
              }}
            >
              Importer document
            </Typography>
            <AddButton
              variant="contained"
              sx={{ minHeight: '43px' }}
              endIcon={<FontAwesome icon={faFileArrowUp} width={18} />}
              disabled={isReadOnly}
            >
              Importez un document
            </AddButton>
          </Grid>
          {observationformFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}
        </Grid>
        <FormContainer sx={{ mt: 1 }} titre="Responsable de l'établissement" isSousElemnt>
          <Grid container spacing={5} sx={{ pt: 1 }}>
            {responsableEtabformFields.map((field) => (
              <Grid
                item
                xs={12}
                sx={field.name === 'civility' ? centerFlexColumn : undefined}
                sm={4}
                key={field.name}
              >
                {renderFieldResponsable(field)}
              </Grid>
            ))}
            <ConditionalComponent isValid={!isReadOnly}>
              <Grid item xs={12}>
                <ActionButtons onSave={handleSubmit} onCancel={onClose} mode={mode} />
              </Grid>
            </ConditionalComponent>
          </Grid>
        </FormContainer>
      </form>
    </>
  );
}
