import * as Yup from 'yup';
import { 
  CustodyCategory, 
  ParticularityTransport, 
  PassengerCivility, 
  SchoolClass, 
  State,
  RelationType,
  WeekType,
  DaysOfWeek,
  SensTrip
} from '@/shared/types/passenger';
import { Civility } from '@/shared/types/common';

export const representativeSchema = Yup.object().shape({
  id: Yup.string(),
  firstName: Yup.string().required('Le prénom est requis'),
  lastName: Yup.string().required('Le nom est requis'),
  email: Yup.string()
    .email('Format email invalide')
    .required('Email est requis'),
  civility: Yup.string()
    .oneOf(Object.values(Civility), 'Civilité invalide'),
  type: Yup.string()
    .oneOf(Object.values(RelationType), 'Type de relation invalide'),
  parentalAuthorization: Yup.boolean(),
  arrivalNotification: Yup.boolean(),
  idCircuit: Yup.string(),
  circuitCode: Yup.string(),
  contacts: Yup.array().of(
    Yup.object().shape({
      landlinePhones: Yup.array().of(
        Yup.string()
          .matches(/^[0-9]+$/, 'Le numéro doit contenir uniquement des chiffres')
          .min(10, 'Le numéro doit contenir au moins 10 chiffres')
          .max(15, 'Le numéro ne doit pas dépasser 15 chiffres')
      ),
      mobilePhones: Yup.array().of(
        Yup.string()
          .matches(/^[0-9]+$/, 'Le numéro doit contenir uniquement des chiffres')
          .min(10, 'Le numéro doit contenir au moins 10 chiffres')
          .max(15, 'Le numéro ne doit pas dépasser 15 chiffres')
      ),
      assignedPersons: Yup.array().of(
        Yup.string()
      )
    })
  ),
  addresses: Yup.array().of(
    Yup.object().shape({
      additionalAddress: Yup.string().required("L'adresse est requise"),
      latitude: Yup.number().required("La latitude est requise"),
      longitude: Yup.number().required("La longitude est requise"),     
    })
  ),
  transportSchedules: Yup.array().of(
    Yup.object().shape({
      weekType: Yup.string()
        .oneOf(Object.values(WeekType), 'Type de semaine invalide')
        .required('Le type de semaine est requis'),
      daySchedules: Yup.array().of(
        Yup.object().shape({
          dayOfWeek: Yup.string()
            .oneOf(Object.values(DaysOfWeek), 'Jour de la semaine invalide')
            .required('Le jour de la semaine est requis'),
          sens: Yup.string()
            .oneOf(Object.values(SensTrip), 'Sens de trajet invalide')
            .required('Le sens de trajet est requis'),
          checked: Yup.boolean()
        })
      )
    })
  )
});

export const otherInterlocutorsSchema = Yup.object().shape({
  civility: Yup.string()
    .oneOf(Object.values(Civility), 'Civilité invalide')
    .required('La civilité est requise'),
  firstName: Yup.string().required('Le nom est requis'),
  lastName: Yup.string().required('Le prénom est requis'),
  mobilePhone: Yup.string().nullable(),
  email: Yup.string().email('Email invalide').required('Email est requis'),
  type: Yup.string()
    .oneOf(Object.values(RelationType), 'Type de relation invalide')
    .required('Le type de relation est requis'),
  observation: Yup.string()
});

export const usagerSchema = Yup.object().shape({
  firstName: Yup.string().required('Le prénom est requis'),
  lastName: Yup.string().required('Le nom est requis'),
  civility: Yup.string()
    .oneOf(Object.values(PassengerCivility), 'Civilité invalide')
    .required('La civilité est requise'),
  email: Yup.string()
    .email('Format email invalide')
    .required('Email est requis'),
  birthDate: Yup.date()
    .nullable()
    .required('La date de naissance est requise')
    .test(
      'birth-date-in-the-past',
      'La date de naissance est invalide : elle doit être une date dans le passé.',
      function (value) {
        if (!value) return false;
        return value < new Date();
      }
    ),
  
  state: Yup.string()
    .oneOf(Object.values(State), 'État invalide')
    .required('L\'état est requis'),
  
  idEstablishment: Yup.number()
    .test('is-greater-than-zero', 'L\'établissement est requis', value => value !== undefined && value > 0)
    .required('L\'établissement est requis'),
  
  idDepartment: Yup.number()
    .test('is-greater-than-zero', 'Le département est requis', value => value !== undefined && value > 0)
    .required('Le département est requis'),
  
  particularityTransport: Yup.string()
    .oneOf(Object.values(ParticularityTransport), 'Particularité de transport invalide'),
  custodyCategory: Yup.string()
    .oneOf(Object.values(CustodyCategory), 'Catégorie de garde invalide'),
  schoolClass: Yup.string()
    .oneOf(Object.values(SchoolClass), 'Classe scolaire invalide'),
  birthPlace: Yup.string(),
  startTransport: Yup.string().required('La date de début du transport est requise'),
  openingDays: Yup.string().nullable(),
  specificity: Yup.string(),
  referentTeacherName: Yup.string(),
  observations: Yup.string(),

  representatives: Yup.array().of(representativeSchema),
  interlocutor: otherInterlocutorsSchema.nullable()
});

export const avenantUsagerSchema = Yup.object().shape({
  dateApplication: Yup.string().required('La date d\'application est requise'),
  codeCircuit: Yup.string().required('Le code circuit est requis'),
  intituleCircuit: Yup.string().required('L\'intitulé du circuit est requis'),
  objetAvenant: Yup.string().required('L\'objet de l\'avenant est requis'),
  specificite: Yup.string().required('La specificite est requis'),
  avenantTemporaire: Yup.string(),
  observation: Yup.string()
});